"use strict";

const MiniSearch = require('minisearch')
const html = require('nanohtml')
const raw = require('nanohtml/raw')

// -- client-side search
// - re-direct form input search to search page w/ querystring as search terms
// - read querystring, run search, populate search page with results

function searchResult() {
  const searchQuery = window.location.search
  const terms = searchQuery.split('=')[1].split('+').join(' ')

  const docs = require('../posts.json')

  // update page-title, search-header, search-input
  const pageTitle = document.querySelector('head title')
  pageTitle.innerText = `Search Results for ${terms}; The Whole Life`

  const searchHeader = document.querySelector('.search-header-term')
  searchHeader.innerText = terms

  const searchInput = document.querySelector('form input')
  searchInput.value = terms
  // --

  const searchTop = document.querySelector('main header')

  // -- search init
  let miniSearch = new MiniSearch({
    fields: ['title.rendered', 'content.rendered', '_embedded', 'contributor'],
    extractField: (document, fieldName) => {

      if (fieldName === '_embedded') {
        const embedded = document['_embedded']
        return [
          embedded['wp:term'][0],
          embedded['wp:term'][1].map(tag => tag.name)
        ]
      }

      if (fieldName === 'contributor') {
        // contributor is an array of objects
        // let's extract [`post_title`, `post_name`] for each object

        const contributor = document['contributor']
        
        return contributor.map(item => {
          return [item.post_title, item.post_name]
        })
      }

      return fieldName.split('.').reduce((doc, key) => doc && doc[key], document)
    },
    storeFields: ['title', 'contributor', 'date', 'excerpt', 'categories', 'slug', '_embedded']
  })

  // index all documents
  miniSearch.addAll(docs)

  let results = miniSearch.search(terms)
 
  if (results.length > 0) {
    const searchHeader = makeIndexSort()
    const searchList = makeSearchResults()

    searchTop.after(searchHeader)
    searchHeader.after(searchList)

    results.map(result => {
      const entry = makeEntry(result)
      searchList.append(entry)
    })

  } else {
    const notFound = makeNotFound()
    searchTop.after(notFound)
  }

}

function makeSearchResults() {
  return html`
    <section id="results" class="bg-mw-copy-wrap pb2 lg-mra lg-mla bg-pr5-2 bg-pb5 lg-pr0 bg-pl5-2 lg-pl0"></section>
  `
}

function makeIndexSort() {
  return html`
    <section class="dn bg-db bg-mw-copy-wrap lg-mra lg-mla pr5-2 lg-pr0 pl5-2 lg-pl0">
      <div id="header-sort-by" data-sortby="title" class="psr x xdr fs-md-small"> 
        <div class="psa ml-2-5 w19p h19p">
          <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M9 0C9.26506 0 9.48193 0.0963849 9.6506 0.289156C9.84337 0.45783 9.93976 0.674699 9.93976 0.93976C9.93976 1.18072 9.84337 1.39759 9.6506 1.59036C9.48193 1.78313 9.26506 1.87952 9 1.87952C8.75904 1.87952 8.54217 1.78313 8.3494 1.59036C8.15663 1.39759 8.06024 1.18072 8.06024 0.93976C8.06024 0.674699 8.15663 0.45783 8.3494 0.289156C8.54217 0.0963849 8.75904 0 9 0ZM5.96386 0.650602C6.22892 0.650602 6.44578 0.746988 6.61446 0.93976C6.80723 1.10843 6.90361 1.3253 6.90361 1.59036C6.90361 1.83133 6.80723 2.04819 6.61446 2.24096C6.44578 2.43373 6.22892 2.53012 5.96386 2.53012C5.72289 2.53012 5.50602 2.43373 5.31325 2.24096C5.12048 2.04819 5.0241 1.83133 5.0241 1.59036C5.0241 1.3253 5.12048 1.10843 5.31325 0.93976C5.50602 0.746988 5.72289 0.650602 5.96386 0.650602ZM12.0361 0.650602C12.3012 0.650602 12.5181 0.746988 12.6867 0.93976C12.8795 1.10843 12.9759 1.3253 12.9759 1.59036C12.9759 1.83133 12.8795 2.04819 12.6867 2.24096C12.5181 2.43373 12.3012 2.53012 12.0361 2.53012C11.7952 2.53012 11.5783 2.43373 11.3855 2.24096C11.1928 2.04819 11.0964 1.83133 11.0964 1.59036C11.0964 1.3253 11.1928 1.10843 11.3855 0.93976C11.5783 0.746988 11.7952 0.650602 12.0361 0.650602ZM2.3494 3.28916C2.3494 3.0241 2.44578 2.80723 2.63855 2.63855C2.83133 2.44578 3.04819 2.3494 3.28916 2.3494C3.55422 2.3494 3.77109 2.44578 3.93976 2.63855C4.13253 2.80723 4.22892 3.0241 4.22892 3.28916C4.22892 3.53012 4.13253 3.74699 3.93976 3.93976C3.77109 4.13253 3.55422 4.22892 3.28916 4.22892C3.04819 4.22892 2.83133 4.13253 2.63855 3.93976C2.44578 3.74699 2.3494 3.53012 2.3494 3.28916ZM13.7711 3.28916C13.7711 3.0241 13.8675 2.80723 14.0602 2.63855C14.253 2.44578 14.4699 2.3494 14.7108 2.3494C14.9759 2.3494 15.1928 2.44578 15.3614 2.63855C15.5542 2.80723 15.6506 3.0241 15.6506 3.28916C15.6506 3.53012 15.5542 3.74699 15.3614 3.93976C15.1928 4.13253 14.9759 4.22892 14.7108 4.22892C14.4699 4.22892 14.253 4.13253 14.0602 3.93976C13.8675 3.74699 13.7711 3.53012 13.7711 3.28916ZM0.650603 5.96385C0.650603 5.69879 0.746988 5.48193 0.939759 5.31325C1.13253 5.12048 1.3494 5.0241 1.59036 5.0241C1.85542 5.0241 2.07229 5.12048 2.24096 5.31325C2.43374 5.48193 2.53012 5.69879 2.53012 5.96385C2.53012 6.20482 2.43374 6.42169 2.24096 6.61446C2.07229 6.80723 1.85542 6.90361 1.59036 6.90361C1.3494 6.90361 1.13253 6.80723 0.939759 6.61446C0.746988 6.42169 0.650603 6.20482 0.650603 5.96385ZM15.4699 5.96385C15.4699 5.69879 15.5663 5.48193 15.759 5.31325C15.9518 5.12048 16.1687 5.0241 16.4096 5.0241C16.6747 5.0241 16.8916 5.12048 17.0602 5.31325C17.253 5.48193 17.3494 5.69879 17.3494 5.96385C17.3494 6.20482 17.253 6.42169 17.0602 6.61446C16.8916 6.80723 16.6747 6.90361 16.4096 6.90361C16.1687 6.90361 15.9518 6.80723 15.759 6.61446C15.5663 6.42169 15.4699 6.20482 15.4699 5.96385ZM0 9C0 8.73494 0.0963852 8.51807 0.289156 8.3494C0.481927 8.15663 0.698795 8.06024 0.939759 8.06024C1.20482 8.06024 1.42169 8.15663 1.59036 8.3494C1.78313 8.51807 1.87952 8.73494 1.87952 9C1.87952 9.24096 1.78313 9.45783 1.59036 9.6506C1.42169 9.84337 1.20482 9.93976 0.939759 9.93976C0.698795 9.93976 0.481927 9.84337 0.289156 9.6506C0.0963852 9.45783 0 9.24096 0 9ZM16.1205 9C16.1205 8.73494 16.2169 8.51807 16.4096 8.3494C16.6024 8.15663 16.8193 8.06024 17.0602 8.06024C17.3253 8.06024 17.5422 8.15663 17.7108 8.3494C17.9036 8.51807 18 8.73494 18 9C18 9.24096 17.9036 9.45783 17.7108 9.6506C17.5422 9.84337 17.3253 9.93976 17.0602 9.93976C16.8193 9.93976 16.6024 9.84337 16.4096 9.6506C16.2169 9.45783 16.1205 9.24096 16.1205 9ZM0.650603 12.0361C0.650603 11.7711 0.746988 11.5542 0.939759 11.3855C1.13253 11.1928 1.3494 11.0964 1.59036 11.0964C1.85542 11.0964 2.07229 11.1928 2.24096 11.3855C2.43374 11.5542 2.53012 11.7711 2.53012 12.0361C2.53012 12.2771 2.43374 12.494 2.24096 12.6867C2.07229 12.8795 1.85542 12.9759 1.59036 12.9759C1.3494 12.9759 1.13253 12.8795 0.939759 12.6867C0.746988 12.494 0.650603 12.2771 0.650603 12.0361ZM15.4699 12.0361C15.4699 11.7711 15.5663 11.5542 15.759 11.3855C15.9518 11.1928 16.1687 11.0964 16.4096 11.0964C16.6747 11.0964 16.8916 11.1928 17.0602 11.3855C17.253 11.5542 17.3494 11.7711 17.3494 12.0361C17.3494 12.2771 17.253 12.494 17.0602 12.6867C16.8916 12.8795 16.6747 12.9759 16.4096 12.9759C16.1687 12.9759 15.9518 12.8795 15.759 12.6867C15.5663 12.494 15.4699 12.2771 15.4699 12.0361ZM2.3494 14.7108C2.3494 14.4458 2.44578 14.2289 2.63855 14.0602C2.83133 13.8675 3.04819 13.7711 3.28916 13.7711C3.55422 13.7711 3.77109 13.8675 3.93976 14.0602C4.13253 14.2289 4.22892 14.4458 4.22892 14.7108C4.22892 14.9518 4.13253 15.1687 3.93976 15.3614C3.77109 15.5542 3.55422 15.6506 3.28916 15.6506C3.04819 15.6506 2.83133 15.5542 2.63855 15.3614C2.44578 15.1687 2.3494 14.9518 2.3494 14.7108ZM13.7711 14.7108C13.7711 14.4458 13.8675 14.2289 14.0602 14.0602C14.253 13.8675 14.4699 13.7711 14.7108 13.7711C14.9759 13.7711 15.1928 13.8675 15.3614 14.0602C15.5542 14.2289 15.6506 14.4458 15.6506 14.7108C15.6506 14.9518 15.5542 15.1687 15.3614 15.3614C15.1928 15.5542 14.9759 15.6506 14.7108 15.6506C14.4699 15.6506 14.253 15.5542 14.0602 15.3614C13.8675 15.1687 13.7711 14.9518 13.7711 14.7108ZM5.96386 15.4699C6.22892 15.4699 6.44578 15.5663 6.61446 15.759C6.80723 15.9277 6.90361 16.1446 6.90361 16.4096C6.90361 16.6506 6.80723 16.8675 6.61446 17.0602C6.44578 17.253 6.22892 17.3494 5.96386 17.3494C5.72289 17.3494 5.50602 17.253 5.31325 17.0602C5.12048 16.8675 5.0241 16.6506 5.0241 16.4096C5.0241 16.1446 5.12048 15.9277 5.31325 15.759C5.50602 15.5663 5.72289 15.4699 5.96386 15.4699ZM12.0361 15.4699C12.3012 15.4699 12.5181 15.5663 12.6867 15.759C12.8795 15.9277 12.9759 16.1446 12.9759 16.4096C12.9759 16.6506 12.8795 16.8675 12.6867 17.0602C12.5181 17.253 12.3012 17.3494 12.0361 17.3494C11.7952 17.3494 11.5783 17.253 11.3855 17.0602C11.1928 16.8675 11.0964 16.6506 11.0964 16.4096C11.0964 16.1446 11.1928 15.9277 11.3855 15.759C11.5783 15.5663 11.7952 15.4699 12.0361 15.4699ZM9 16.1205C9.26506 16.1205 9.48193 16.2169 9.6506 16.4096C9.84337 16.5783 9.93976 16.7952 9.93976 17.0602C9.93976 17.3012 9.84337 17.5181 9.6506 17.7108C9.48193 17.9036 9.26506 18 9 18C8.75904 18 8.54217 17.9036 8.3494 17.7108C8.15663 17.5181 8.06024 17.3012 8.06024 17.0602C8.06024 16.7952 8.15663 16.5783 8.3494 16.4096C8.54217 16.2169 8.75904 16.1205 9 16.1205Z" fill="#212121"/></svg>
        </div>
        <div id="sort-by-title" data-sort="title" class="curp md-w25 pb0-5 bb0-75-dark-green">Title<span class="pl0-5">${raw('&darr;')}</span></div>
        <div id="sort-by-contributors" data-sort="contributors" class="curp md-w25 pb0-5 bb0-75-dark-green">Author(s) / Producer(s)<span class="dn pl0-5">${raw('&darr;')}</span></div>
        <div id="sort-by-date" data-sort="date" class="curp md-w16-67 pb0-5 bb0-75-dark-green">Date<span class="dn pl0-5">${raw('&darr;')}</span></div>
        <div class="md-w33 pb0-5 bb0-75-dark-green">Description</div>
      </div>
    </section>
  `
}

function makeNotFound() {
  return html`
    <div class="tb-mw-copy-wrap w100 tb-mw-copy-post bg-mw100 lg-mra lg-mla bg-pr5-2 lg-pr0 bg-pl5-2 lg-pl0 bg-pb5">
      <p>Nothing was found with this search term.</p>
    </div>
  `
}

function makeIcon(cats) {
  const index = {
    'contributor': html`<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M17.625 9C17.625 13.7694 13.7693 17.625 9 17.625C4.23139 17.625 0.375 13.7436 0.375 9C0.375 4.23063 4.23067 0.375 9 0.375C13.7693 0.375 17.625 4.23063 17.625 9Z" fill="#FFE8D3" stroke="#212121" stroke-width="0.75"/>
</svg>`,
    'keyword': html`<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M9 18C13.9764 18 18 13.9765 18 9C18 4.02352 13.9764 0 9 0C4.02357 0 0 4.02352 0 9C0 13.95 4.02357 18 9 18Z" fill="#6A7880"/>
</svg>`,
    'mixed-media': html`<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M17.625 9C17.625 13.7694 13.7693 17.625 9 17.625C4.23139 17.625 0.375 13.7436 0.375 9C0.375 4.23063 4.23067 0.375 9 0.375C13.7693 0.375 17.625 4.23063 17.625 9Z" fill="#FFD3E8" stroke="#FF6161" stroke-width="0.75"/>
</svg>`,
    'photo-video': html`<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M10 19C14.9764 19 19 14.9765 19 10C19 5.02352 14.9764 1 10 1C5.02357 1 1 5.02352 1 10C1 14.95 5.02357 19 10 19Z" fill="#FFFCE2" stroke="#FFBBB2" stroke-width="0.75"/>
</svg>`,
    'reference': html`<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M17.625 9C17.625 13.7694 13.7693 17.625 9 17.625C4.23139 17.625 0.375 13.7436 0.375 9C0.375 4.23063 4.23067 0.375 9 0.375C13.7693 0.375 17.625 4.23063 17.625 9Z" fill="#D3FAFF" stroke="#6A7880" stroke-width="0.75"/>
</svg>`,
    'sound': html`<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M17.625 9C17.625 13.7694 13.7693 17.625 9 17.625C4.23139 17.625 0.375 13.7436 0.375 9C0.375 4.23063 4.23067 0.375 9 0.375C13.7693 0.375 17.625 4.23063 17.625 9Z" fill="#E5E5E5" stroke="#583DC5" stroke-width="0.75"/>
</svg>`,
    'text': html`<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M17.625 9C17.625 13.7694 13.7693 17.625 9 17.625C4.23139 17.625 0.375 13.7436 0.375 9C0.375 4.23063 4.23067 0.375 9 0.375C13.7693 0.375 17.625 4.23063 17.625 9Z" fill="#CAEFD0" stroke="#889AA4" stroke-width="0.75"/>
</svg>`
  }

  return index[cats[0]['slug']]
}

function makeContributors(contributors) {
  return contributors.map((contributor, idx) => {
    const hasComma = contributors.length -1 === idx ? '' : ','

    return html`
        <a href="../../../${contributor[1]}/index.html" class="tdn fgc-lait-green-hv">${contributor[0]}${hasComma}</a>
      `
  })
}

function makeDate(date) {
  return new Date(date).toLocaleDateString('nl-NL', {
    timeZone: 'Europe/Berlin',
    day: '2-digit',
    month: '2-digit',
    year: 'numeric',
  }).replace(/-/g, '.')
}

function makeEntry(result) {
  return html`
    <div class="bg-db-child-hv w100 tb-mw-copy-post bg-mw100 x xdc bg-xdr xw td-xnw pt1 psr">
      <div class="bg-child-hv bg-dn psa t0 r0 bg-ra bg-l0 pt2-9 bg-pt1-2 bg-ml-2-5 xo3 bg-xo1">
      ${makeIcon(result['_embedded'][0])}
      </div>
      <div data-type="title" class="w100 bg-w25 pr1-5 pb0-5 bg-pb1 bg-bb0-75-dash-dark-green xo2">
        <a href="../${result.slug}/index.html" class="ft-barlow fgc-lait-green-hv fs-md-h2 bg-fs-md-h3 tdn">${raw(result.title.rendered)}</a>
      </div>
      <div data-type="contributors" class="w100 bg-w25 pr0-5 pb1 fs-sm-small bg-fs-md-small bg-bb0-75-dash-dark-green xo4 bg-xo3">
        ${makeContributors(result.contributor)}
      </div>
      <div data-type="date" class="w100 bg-w16-67 pr0-5 pb0-5 bg-pb1 fs-sm-small bg-fs-md-small bg-bb0-75-dash-dark-green xo1 bg-xo5">
        ${makeDate(result.date)}
      </div>
    <div data-type="description" class="w100 bg-w33 pb1 fs-sm-small bg-fs-md-small bb0-75-dash-dark-green xo5 bg-xo6">${raw(result.excerpt.rendered)}</div>
</div>
  `
}

searchResult()

