const sessionDB = require('./session-db')
const {mapListHeightiOS}  = require('./browser-specific-rules')
const PDFObject = require('pdfobject')

mapListHeightiOS()

// set correct image-slideshow margins to first and list item
function setMargins(gallery) {
  let spacing = document.querySelector('main header')
  if (spacing === null) {
    return
  }
  
  let spacingGBR = spacing.getBoundingClientRect()
  let spacingText = spacing.firstElementChild
  let spacingTextGBR = spacingText.getBoundingClientRect()
  let mainGBR = document.querySelector('main').getBoundingClientRect()

  let mainPadding = spacingTextGBR.left - mainGBR.left
  let mainMarginEnd = (mainGBR.right - spacingGBR.right) + mainPadding

  if (gallery.length > 0) {
    Array.from(gallery).map(g => {

      const galleryWrapper = g.parentNode

      // remove img-normal to parent element as it breaks
      // the gallery design
      galleryWrapper.classList.remove('img-normal')

      if (g.childElementCount > 0) {
        let imageFirst = g.firstElementChild

        // Wordpress adds fig-caption as last item in the list of images
        // make sure to grab the correct last-image
        let imageLast = g.lastElementChild.nodeName === "FIGCAPTION"
            ? g.childNodes[g.childNodes.length -2] 
            : g.lastElementChild

        // let caption = g.querySelector('figcaption')

        if (window.innerWidth >= 1024) {
          g.style.paddingLeft = `${mainPadding}px`

          // macOS Safari has a bug (?) for which it can't correctly add
          // the paddingRight to the imageLast element *and* pushing the
          // element to the left (so to add some empty space at the end it) 
          imageLast.style.marginRight = `${mainMarginEnd}px`

        } else {
          g.removeAttribute('style')
        }

      }
    })
  } 
  
}

// --

// -- setup footnotes
// refs:
// - <https://ia.net/wp-content/themes/iA-library/main.js?ver=5.4.8>
// - <https://stackoverflow.com/a/49860927>

function footnotesCleanup(article) {
  let sups = []
  let refs = []

  if (article !== null) {
    sups = Array.from(article.querySelectorAll('.fs-copy-sb sup'))
  } else {
    sups = Array.from(document.querySelectorAll('.fs-copy sup'))
  }

  Array.from(sups).map((sup, i) => {

    // replace sub text styling
    const link = sup.children[0]
    link.textContent = link.textContent.replace('[', '').replace(']', '')
    
    // set sup numbering based on list of footnotes number
    let idx = i + 1
    link.textContent = idx

    // update URLs and set IDs
    const baseURL = `${window.location.origin}${window.location.pathname}`

    const supbaseURL = link.href.split('-').slice(0, -1).join('-')
    const orURL = `${supbaseURL}${idx}`
    link.href = orURL

    const refURLs = Array.from(document.querySelectorAll(`a[href='..${window.location.pathname}#footnote-ref-${idx}']`))

    // stupid code but first add all found refs
    // then use array length to set correct numbering to them
    if (refURLs.length > 0) {
      refURLs.map(refURL => { 
        refs.push(refURL)
      })
    }

    refs.map((ref, i) => {
      let cidx = i +1
      ref.href = `${baseURL}#footnote-ref-${cidx}`

      ref.parentNode.setAttribute('id', `footnote${cidx}`)
      ref.parentNode.classList.add('fs-md-small')

      ref.parentNode.parentNode.classList.add('ft-marker')
      ref.parentNode.parentNode.parentNode.parentNode.classList.add('footnotes-wrap')
    })

    // fix up sup after all the rest is set
    sup.setAttribute('id', `footnote-ref-${idx}`)
    link.href = `${baseURL}#footnote${idx}`
  })

  // add custom on-click event to add scrolling with offset
  if (window.innerWidth <= 1400) {
    sups.forEach((sup, i) => {

      sup.addEventListener('click', (e) => {
        e.preventDefault()

        let idx = i +1

        const clickHash = e.target.hash.substr(1)

        const link = sup.children[0]
        const supbaseURL = link.href.split('/').slice(0, -1).join('/')
        const orURL = `${supbaseURL}/#footnote-ref-${idx}`

        let target = document.querySelector(`a[href='${orURL}']`)
        if (target !== null) {
          target = target.parentNode

          const targetPos = target.getBoundingClientRect().top + window.scrollY

          const wpadminbarHeight = document.querySelector('#wpadminbar') !== null
                ? document.querySelector('#wpadminbar').getBoundingClientRect().height
                : 0

          const header = document.querySelector('.nav-header')
          let headerHeight = 0
          if (header !== null) {
            headerHeight = header.getBoundingClientRect().height
          }

          let offsetPosition = targetPos - wpadminbarHeight - headerHeight

          if (article !== null) {
            target.scrollIntoView({behavior: 'smooth'})

          } else {

            window.scroll({
              top: offsetPosition,
              left: 0,
              behavior: 'smooth'
            });

            window.history.pushState(null, '', `#${clickHash}`)
          }

        }
      })

    })
  }


  // add custom on-click event to add scrolling with offset
  if (refs.length > 0) {
    refs.forEach(ref => {

      let header = document.querySelector('.nav-header')
      let headerHeight = 0
      if (header !== null) {
        headerHeight = header.getBoundingClientRect().height
      }

      ref.addEventListener('click', (e) => {
        e.preventDefault()

        const clickHash = e.target.hash.substr(1)
        let target = document.querySelector(`#${clickHash}`)

        if (target !== null) {
          target = target.parentNode

          const targetPos = target.getBoundingClientRect().top + window.scrollY

          const wpadminbarHeight = document.querySelector('#wpadminbar') !== null
                ? document.querySelector('#wpadminbar').getBoundingClientRect().height
                : 0

          let offsetPosition = targetPos - wpadminbarHeight - headerHeight

          if (article !== null) {
            target.scrollIntoView({behavior: 'smooth'})

          } else {
            window.scroll({
              top: offsetPosition,
              left: 0,
              behavior: 'smooth'
            });

            window.history.pushState(null, '', `#${clickHash}`)
          }
        }
      })

    })
  }

  return {sups, refs}
  
}

function fixLandingFootnote() {
  // check if URL has #footnote<>
  if (window.location.hash !== '') {
    
    let hash = window.location.hash
    if (hash.startsWith('#footnote-ref-')) {
      return
    } else if (hash.startsWith('#footnote')) {
      // adjust window scroll to include fixed header height

      let header = document.querySelector('.nav-header')
      let headerHeight = 0
      if (header !== null) {
        headerHeight = header.getBoundingClientRect().height
      }

      document.documentElement.scrollTop -=  headerHeight
    }
  }
}

function setFootnotes(footnotes, sups, isFullArticle) {

  let footnotesWrap = document.querySelector('.footnotes-wrap')
  if (footnotesWrap === null) {
    // article has not footnotes
    return
  }
  
  if (isFullArticle && window.innerWidth >= 1400) {
    let bodyRect = document.body.getBoundingClientRect()
    let contentBlocksWrapper = document.querySelector('.content-blocks-wrapper')
    let contentBlocksWrapperRect = contentBlocksWrapper.getBoundingClientRect()
    let contentBlock = contentBlocksWrapper.querySelector('.post-block-wrap')
    let contentBlockRect = contentBlock.getBoundingClientRect() 

    let footnotesWrapRect = footnotesWrap.getBoundingClientRect()

    footnotesWrap.style.position = 'absolute'
    footnotesWrap.style.top = `${contentBlocksWrapperRect.top}px`
    footnotesWrap.style.right = `${bodyRect.right - contentBlockRect.right - 20}px`
    footnotesWrap.style.height = '100%'

    sups.forEach((sup, i) => {
      // hide footnotes link scrolling, as unnecessary
      sup.children[0].classList.add('pen')
      footnotes[i].classList.add('dn')

      let top = Math.round(sup.getBoundingClientRect().top - footnotesWrap.getBoundingClientRect().top)
      let note = footnotes[i].parentNode
      if (note.previousElementSibling) {
        top = solveFor(note, top, footnotesWrap.getBoundingClientRect().top)
      }

      note.style.position = 'absolute'
      note.style.top = `${top}px`
      note.classList.add('tb-pr1')

    })
    
  } else {

    // display footnotes link scrolling, as necessary
    sups.forEach((sup, i) => {
      sup.children[0].classList.remove('pen')

      // if we set footnotes wrongly in the post, let's check that
      // each ref has an sup element, eg that both footnotes and sups have
      // the same length
      if (footnotes[i] !== undefined) {
        footnotes[i].classList.remove('dn')
        let note = footnotes[i].parentNode
        note.removeAttribute('style')
      }
    })

    const footnotesWrap = document.querySelector('.footnotes-wrap')
    footnotesWrap.removeAttribute('style')

    if (!isFullArticle) {
      footnotesWrap.classList.remove('footnotes-wrap')
    }

  }
}

function solveFor (el, currentTop, parentTop) {
  const previousEl = el.previousElementSibling
  const previousElTop = parseInt(previousEl.style.top, 10)
  const previousElBottom = previousEl.getBoundingClientRect().bottom - parentTop

  // check for overlaps
  if (currentTop === previousElTop) {
    const newTop = currentTop + previousEl.getBoundingClientRect().height
    return newTop
  } else if (currentTop < previousElBottom) {
    const newTop = previousElBottom
    return newTop
  } else {
    return currentTop
  } 

}

// -- image-gallery: allow images to be dragged
//    eg, to do the same action as if scrolling
//    (they wanted to hide the scrollbar, which would have
//    allowed this action)
// <https://htmldom.dev/drag-to-scroll/>

function dragToScroll(target) {
  target.style.cursor = 'grab'

  let pos = {top: 0, left: 0, x: 0, y: 0}

  const mouseDownHandler = function (e) {
    // add cursor styles
    target.style.cursor = 'grabbing'
    target.style.userSelect = 'none'

    // left, top: current scroll
    // x, y: get current mouse position
    pos = {
      left: target.scrollLeft,
      top: target.scrollTop,
      x: e.clientX,
      y: e.clientY
    }

    document.addEventListener('mousemove', mouseMoveHandler)
    document.addEventListener('mouseup', mouseUpHandler)
  }

  const mouseMoveHandler = function (e) {
    // how far the mouse has been moved
    const dx = e.clientX - pos.x
    const dy = e.clientY - pos.y

    // scoll element
    target.scrollTop = pos.top - dy
    target.scrollLeft = pos.left - dx
  }

  const mouseUpHandler = function () {
    target.style.cursor = 'grab'
    target.style.removeProperty('user-select')

    document.removeEventListener('mousemove', mouseMoveHandler)
    document.removeEventListener('mouseup', mouseUpHandler)
  }

  target.addEventListener('mousedown', mouseDownHandler)

}

function embedPDF() {

  var options = {
    fallbackLink: "<p>This is a <a href='[url]'>fallback link</a> for the PDF.</p>",
  };

  const docs = Array.from(document.querySelectorAll('.pdf-embed'))
  if (docs.length > 0) {
    docs.map(doc => {
      const URL = doc.dataset.mediaurl
      PDFObject.embed(URL, doc, options) 
    })
  }

}


// -- run everything
window.onload = () => {
  // check if mapList exists,
  // if not, build it for the reader article page
  const isPostURL = window.location.pathname !== '/'
        && window.location.pathname.length > 1

  if (isPostURL) {
    sessionDB.init()

    const gallery = Array.from(document.querySelectorAll('.wp-block-gallery'))
          // .filter(gal => {
          //   if (!gal.parentNode.classList.contains('img-normal')) {
          //     return gal
          //   }
          // })

    setMargins(gallery)

    gallery.map(target => {
      dragToScroll(target)
    })

    // -- add footnotes to the side of screen size if wide enough
    const {sups,refs} = footnotesCleanup(null)
    fixLandingFootnote()
    setFootnotes(refs, sups, true)

    window.addEventListener('resize', () => {
      setMargins(gallery)
      let i = 0
      setFootnotes(refs, sups, true)
    })
  }

  embedPDF()

}

module.exports = {footnotesCleanup, setFootnotes}
