"use strict";

const html = require('nanohtml')
const raw = require('nanohtml/raw')
const fetchPOST = require('../fetch').post
const slugify = require('slugify')
const {create_html_tag_label} = require('./html-layer')
const {transform_tag} = require('./fetch-data')
const {get_camera_center} = require('./camera')

const {generate_tag_event_handlers} = require("./interactions.js")

function addTag(svg, isPreview, tags, articles, group, mapSize) {

  const box = html`
    <form method="POST" onsubmit=${onSubmit} class="bgc-white psf z3 t50 l50 ttcc mw-fourth pt0-5 pr0-5 pb0-5 pl0-5">
      <div class="tc-box-wrap x xjb pb0-5">
        <p class="tc-box-text fs-md-small">Add a new tag</p>
        <button type="button" onclick=${onClose} class="tc-box-close psr button-s">
          <span class="curp psa t50 l50 ttcc fs-md-small">${raw('&#10005;')}</span>
        </button>
      </div>
      <input type="text" value="" placeholder="Type a word" required="required" class="tc-box-input mr0-5 fs-md-small">
      <button type="submit" class="tc-box-add button-s">Add</button>
    </form>
  `


  async function onSubmit (e) {
    e.preventDefault()
    const tagValue = e.target[1].value

    try {
      const camera_coord = get_camera_center()

      const resp = await fetchPOST('/wp-json/wp/v2/tags', {
        description: '',
        name: tagValue,
        slug: slugify(tagValue).toLowerCase(),
        map_location: [camera_coord.x, camera_coord.y]
      })

      if ('data' in resp && resp.data.status !== 200) {
        displayError(resp)
      } else {

        const tag_added = transform_tag(resp)
        tags.push(tag_added)

        await refreshMap(tag_added, group, mapSize)
      }

    } catch (err) {
      console.log('tag-create err =>', err)
      displayError(null)
    }
  }

  function onClose () {
    box.remove()
  }

  const boxClose = box.querySelector('.tc-box-close')
  const boxText = box.querySelector('.tc-box-text')
  const boxInput = box.querySelector('.tc-box-input')
  const boxWrap = box.querySelector('.tc-box-wrap')
  const boxAdd = box.querySelector('.tc-box-add') 

  async function refreshMap(tag_added, group) {
    box.remove()

    const mapUI = document.querySelector('#map-ui')

    // select newly added tag to DOM and set bright color to it
    create_svg_tag(group, tag_added, tags, articles, true, mapSize)
    create_html_tag_label(mapUI, mapSize, tag_added, isPreview, tags, articles)

    // set new tag to bright color (pink)
    // html layer
    tag_added.html_element.children[0].style.backgroundColor = '#FFBBB2'

    // svg layer
    tag_added.svg_elements[0].setAttribute('fill', '#FFBBB2')
  }

  function displayError(data) {
    boxInput.remove()
    boxAdd.remove()
    boxClose.remove()

    const wrap = html`
      <div class="tc-retry-wrap tac">
        <button type="button" onclick=${onRetry} class="button-s">Retry</button>
      </div>
    `
    box.append(wrap)
 
    if (data !== null) {
      boxText.innerHTML = data.message
    } else {
      boxText.innerHTML = "An error happened. Try again."
    }
  }

  function onRetry () {
    boxClose.remove()
    box.querySelector('.tc-retry-wrap').remove()

    boxText.innerHTML = "Add a new tag"
    boxInput.value = ""

    boxWrap.append(boxClose)
    box.append(boxInput)
    box.append(boxAdd)
  }

  document.querySelector('main').append(box)

}

function create_svg_tag (svg, tag, tags, articles, isPreview, mapSize) {
  const {onClick, onHoverIn, onHoverOut} = generate_tag_event_handlers(tag, tags, articles, mapSize)

  let dot = svg.circle(tag.map_location)
      .radius(10)
      .fill('#6A7880')
      .addClass(`node-map node-tag circle-${tag.id} curp`)
      .setAttributes({'data-id': tag.id})
  
  if (!isPreview) {
    dot.addEventListener('click', onClick)
    dot.addEventListener('mouseover', onHoverIn)
    dot.addEventListener('mouseout', onHoverOut)
  }

  tag.svg_elements.push(dot)
}

module.exports = addTag
