const sessionDB = require('../session-db')
const {mapListBoxAppend} = require('../map-list-box')
const fetchPUT = require('../fetch').put
const {start_panning, stop_panning,
       get_map_position, get_cursor_position,
       get_camera_zoom, get_camera_zoom_reverse,
       on_pointer_up} = require('./camera')
const {select_article, select_tag,
       higlight_article_connections, highlight_tag_connections, clear_higlights} = require('./map-node-select')
const {transform_tag} = require('./fetch-data')
const compute_force_directed_layout = require('./force-directed-layout')
const {update_html_layer, update_html_article_box} = require('./update-html-layer')
const {origami, origamiUpdateColor} = require('./origami')
const {navToggleMap, navToggleSplit} = require('../nav-toggle-buttons')
const html = require('nanohtml')
const raw = require('nanohtml/raw')
const {getNodeColor} = require('./map-node-select') 
const mapNodeReset = require('./map-node-reset')
const mapBoxRemoveItem = require('../map-list-box').itemDelete
const updateSplitReader = require('./split-reader-update')
const {mapNavNotify} = require('../map-nav-toggle')


let isDraggingTarget = null
let cursorDown = {x: 0, y: 0}
let shift = {x: 0, y: 0}

document.addEventListener('mousemove', (e) => {
  if (isDraggingTarget !== null) {
    onMouseMove(e, isDraggingTarget, shift)
  }
})

function onMouseMove(e, target, shift) {
  node_preview_remove(target)

  const cursor = get_cursor_position(e.x, e.y)
  const cursor_map_position = get_map_position(cursor.x, cursor.y) 

  target.map_location = ear.vector(cursor_map_position.x - shift.x, cursor_map_position.y - shift.y)

  update_html_article_box(target)
}

function generate_article_event_handlers(mapSize, article, articles, tags) {

  let onMouseDown = (e) => {
    clear_higlights()
    node_preview_remove(article)

    isDraggingTarget = article
    article.html_element.classList.add('z3')

    const cursor = get_cursor_position(e.x, e.y)
    cursorDown = cursor
    const cursor_map_location = get_map_position(cursor.x, cursor.y)  
    const article_map_location = article.map_location

    shift.x = cursor_map_location.x - article_map_location.x
    shift.y = cursor_map_location.y - article_map_location.y
  }

  let onMouseUp = (e) => { 
    const currentCursor = get_cursor_position(e.x, e.y)

    let source_node = article 
    let target_node = null

    // if `on-mouse-move` happened after `on-mouse-down`
    // isDraggingTarget is (still) an article (eg not `null`),
    // in which case we avoid to select the node (enlarge it)
    if (isDraggingTarget !== null
        && isDraggingTarget.id === article.id
        && currentCursor.x !== cursorDown.x
        || isDraggingTarget !== null && currentCursor.y !== cursorDown.y) {
      
      isDraggingTarget.html_element.classList.remove('z3')
      isDraggingTarget = null

      // trigger map-layout-change on node drag-and-release
      let selected_nodes = [source_node.id]
      if (target_node !== null) {
        selected_nodes.push(target_node.id)
      }
      
      compute_force_directed_layout(articles, selected_nodes)
      update_html_layer(tags, articles, false)

      return
    }

    // check what we clicked, allow to add node to the list if:
    // - nodeSelected is null (fresh page load, reload, etc)
    // - nodeSelected is not already part of db.articles 
    let db = sessionDB.getAll()
    const isAdded = sessionDB.isAdded(db, article.id)
    if (db.nodeSelected.id === null && isAdded === false || isAdded === false) {

      isDraggingTarget.html_element.classList.remove('z3')
      isDraggingTarget = null

      db.nodeSelected = {id: article.id, type: article.type}
      sessionDB.save(db)

      if (!article.html_element.classList.contains('ts1-25')) {
        select_article(article) 

        // update origami color using article's category color
        const origamiGroup = document.querySelector('#origami-group-ui')
        Array.from(origamiGroup.children).map(child => {
          child.remove()
        })

        const category = article.categories !== undefined
              ? article.categories[0].slug
              : 'default'

        let origamiOpacity = category === 'sound' ? 0.2 : 0.5 
        origami(origamiGroup, mapSize, origamiOpacity, true)
        origamiUpdateColor(article)

      } else {
        nodeDeselect(article, tags, articles)
      }

      let mapResultBox = document.querySelector('#map-result-box')
      mapListBoxAppend(article, mapResultBox, false, articles, tags, true)

      updateSplitReader(tags, articles)

      mapNavNotify()

    } else {

      isDraggingTarget.html_element.classList.remove('z3')
      isDraggingTarget = null

      db.nodeSelected = {id: article.id, type: article.type}
      sessionDB.save(db)

      if (!article.html_element.classList.contains('ts1-25')) {
        select_article(article)

        // update origami color using article's category color
        const origamiGroup = document.querySelector('#origami-group-ui')
        Array.from(origamiGroup.children).map(child => {
          child.remove()
        })

        const category = article.categories !== undefined
              ? article.categories[0].slug
              : 'default'

        let origamiOpacity = category === 'sound' ? 0.2 : 0.5 
        origami(origamiGroup, mapSize, origamiOpacity, true)
        origamiUpdateColor(article)
        
      } else {
        // this update split-box-list by removing an item
        nodeDeselect(article, tags, articles)
      }

      updateSplitReader(tags, articles)

      mapNavNotify()
    }

  }

  let onHoverIn = () => {
    if (!hasTouch()) {
      higlight_article_connections(article)
    }
  }

  let onHoverOut = () => {
    if (!hasTouch()) {
      clear_higlights()
      node_preview_remove(article)
    }
  } 

  let onMove = () => {
    let db = sessionDB.getAll()
    const isSelected = db.articles.find(art => art.id === article.id)

    if (!isSelected) {
      node_preview_add(article)
    }
  }

  return {onHoverIn, onHoverOut,
          onMove,
          onMouseDown, onMouseUp
         }
}

function node_preview_add(article) {
  let preview = document.querySelector(`[data-id="${article.id}"]`)
      .querySelector('.node-preview')

  if (preview !== null) {
    preview.classList.remove('pen')
    preview.style.opacity = 1

    preview.style.width = `180px`

    let zoom_factor = get_camera_zoom()
    let zoomReverse = get_camera_zoom_reverse()
    let boundingBox = article.html_element.getBoundingClientRect()

    preview.style.left = `${(boundingBox.width - (boundingBox.width / 4)) / zoom_factor}px`
    preview.style.top = `${(boundingBox.height - (boundingBox.height / 4)) / zoom_factor}px`

    // note: applying zoom-reverse to the preview-box keep its proportions
    // intact across zoom change, but shifts it off from the correcty x,y
    // position it should stay (proportionally speaking)
    // we add transform-origin: top left to fix this
    preview.style.transformOrigin = 'top left'
    preview.style.transform = `scale(${zoomReverse})`
  }
}

function node_preview_remove(article) {
  let preview = document.querySelector(`[data-id="${article.id}"]`)
      .querySelector('.node-preview')

  if (preview !== null) {
    preview.style.opacity = 0
    preview.classList.add('pen')
  }
}

function generate_tag_event_handlers(tag, tags, articles, mapSize) {
  let onClick = (e) => {
    let db = sessionDB.getAll()

    const isAdded = sessionDB.isAdded(db, tag.id)
    if (db.nodeSelected.id === null && isAdded === false || isAdded === false) {

      db.nodeSelected = {id: tag.id, type: tag.type}
      sessionDB.save(db)

      if (!tag.html_element.classList.contains('ts1-25')) {
        select_tag(tag)

        tag.articles.map(article => {
          select_article(article)
        }) 

      } else {
        nodeDeselect(tag, tags, articles)
      }

      // update origami color using tag's category color
      const origamiGroup = document.querySelector('#origami-group-ui')
      Array.from(origamiGroup.children).map(child => {
        child.remove()
      })

      let origamiOpacity = 0.2
      origami(origamiGroup, mapSize, origamiOpacity, true)
      origamiUpdateColor(tag)

      // store in map list box
      const mapResultBox = document.querySelector('#map-result-box')
      mapListBoxAppend(tag, mapResultBox, false, articles, tags, true)

      updateSplitReader(tags, articles)

      mapNavNotify()

    } else {

      db.nodeSelected = {id: tag.id, type: tag.type}
      sessionDB.save(db)

      if (!tag.html_element.classList.contains('ts1-25')) {
        select_tag(tag)

        tag.articles.map(article => {
          select_article(article)
        })
        
      } else {
        nodeDeselect(tag, tags, articles)
      }

      // update origami color using article's category color
      origamiUpdateColor(tag)
      updateSplitReader(tags, articles)

      mapNavNotify()
    }
  }

  let onHoverIn = () => {
    if (!hasTouch()) {
      highlight_tag_connections(tag)
    }
  }

  let onHoverOut = () => {
    if (!hasTouch()) {
      clear_higlights()
    }
  }

  return {onClick, onHoverIn, onHoverOut}
}

function nodeDeselect(nodeSelected, tags, articles) {
  const mapResultBox = document.querySelector('#map-result-box')
  const mapResultHeader = mapResultBox.querySelector('header')
  const mapResultList = mapResultBox.querySelector('#map-result-list') 

  let db = sessionDB.getAll()

  function deselect(node) {
    mapNodeReset(node, false)
    sessionDB.removeItemAll(node.id) 

    mapBoxRemoveItem(node, mapResultList, mapResultHeader, false, false, articles, tags)
  }

  // tag: remove also all connected articles
  if (nodeSelected.type === 'tag') {
    const tag = tags.find(tag => tag.id === nodeSelected.id)
    tag.articles.map(article => {
      deselect(article)
    })
  }

  deselect(nodeSelected)
 
  updateSplitReader(tags, articles)
}

// --- map-preview
function move_tag_on_map(svg, tags) {
  let tagSelected = null

  // ---
  svg.onPress = async(e) => {

    if (e.target.nodeName === 'circle' || e.target.dataset.type === 'keyword') {
      tagSelected = tags.find(tag => tag.id === Number(e.target.dataset.id))
    }

  }

  svg.onMove = (e) => {
    let map_position = null
    map_position = get_map_position(e.x, e.y)

    // show x,y on map
    const mapCoord = document.querySelector('#map-coord')
    if (mapCoord !== null) {
      let map_positionX = toFixed(map_position.x, 2)
      let map_positionY = toFixed(map_position.y, 2)
      mapCoord.innerHTML = `x: ${map_positionX}, y: ${map_positionY}`
    }

    function toFixed(num, fixed) {
      let re = new RegExp('^-?\\d+(?:\.\\d{0,' + (fixed || -1) + '})?');
      return num.toString().match(re)[0];
    }

    // checking for `tagSelected === null` prevents onMove to trigger
    // when we drag quickly a node /:
    if (tagSelected === null) {
      return
    }

    // -- drag node around
    stop_panning()

    if (map_position !== null) {
      if (e.buttons === 0) {
        return
      }

      // could update `tagSelected.map_location` instead?
      tagSelected.svg_elements[0].setPosition(map_position.x, map_position.y)

      const LABEL_OFFSET = 40
      let boundingBox = tagSelected.html_element.getBoundingClientRect()
      tagSelected.html_element.style.left = `${map_position.x - boundingBox.width/2}px`
      tagSelected.html_element.style.top = `${map_position.y - LABEL_OFFSET}px`
    }

  }

  svg.onRelease = async(e) => {
    if (tagSelected === null) {
      return
    }

    // when a dot is released, we want to update the tag x,y to the new values
    // POST /wp/v2/tags/<id>
    const tagID = Number(tagSelected.svg_elements[0].dataset.id)
    const map_location = [Number(tagSelected.svg_elements[0].getAttribute('cx')),
                          Number(tagSelected.svg_elements[0].getAttribute('cy'))];

    // we must pass tagID both in URL as well as in body
    try {
      const t = await fetchPUT(`/wp-json/wp/v2/tags/${tagID}`, {
        id: tagID,
        map_location: map_location
      })

      // check if we have added a new node, in case push it to points[]
      // if not, we created a new tag
      const tagNew = tags.find(point => point.id === Number(tagSelected.svg_elements[0].dataset.id))

      if (tagNew === undefined) {
        tags.push(t)
      } else {
        // we manually update dragged node map-location x,y
        tagNew['map_location'] = map_location
      }


      // reset selection,
      // re-enable map-panning by setting is-pointer-up to false
      // (so no panning starts right away)
      tagSelected = null
      on_pointer_up(e)
      start_panning()

    } catch (err) {
      console.log('fetchPUT, catch err (what to do?) =>', err)
    } 
  } 

}

function hasTouch() {
  // <https://stackoverflow.com/questions/23885255/how-to-remove-ignore-hover-css-style-on-touch-devices>
  return 'ontouchstart' in document.documentElement
    || navigator.maxTouchPoints > 0
    || navigator.msMaxTouchPoints > 0;
}


module.exports = {generate_article_event_handlers,
                  generate_tag_event_handlers,
                  move_tag_on_map,
                  updateSplitReader}
