const {splash, splashPost} = require('../splash')
const {fetch_articles, fetch_tags} = require('./fetch-data')
const {create_svg_layer} = require('./svg-layer')
const {create_html_layer} = require('./html-layer')
const {init_camera, set_camera_position} = require('./camera')
const {select_article, select_tag} = require('./map-node-select.js')
const {navToggleSwitch} = require('../nav-toggle-switch')
const mapHelp = require('../map-help')
const {mapListBox} = require('../map-list-box')
const expandInReaderPage = require('../expand-in-reader-page')
const {splitButtonClick} = require('../nav-toggle-switch')
const mapListBoxToolbar = require('../map-list-box-toolbar')
const splitBoxToolbar = require('../split-box-toolbar')
const sessionDB = require('../session-db')
const compute_force_directed_layout = require('./force-directed-layout')
const {update_html_layer} = require('./update-html-layer')
const {origami} = require('./origami')
const {mapNavToggle, viewToggle} = require('../map-nav-toggle')
const {browserSpecificRules, mapListHeightiOS}  = require('../browser-specific-rules')
const {keepPositionWithinMapBoundaries} = require('./camera')

browserSpecificRules()
mapListHeightiOS()

async function init_map(isPreview) {
  const mapSize = {width: 8000, height: 8000}

  let origamiSplashGroup
  if (!isPreview) {
    // -- setup intro box, unless shown already (and tracked in session-db)
    origamiSplashGroup = splash(mapSize) 
  }

  // load all data from Wordpress API
  let tags = await fetch_tags(isPreview, null)
  let articles = await fetch_articles(tags, null)

  // create SVG & HTML element layers
  create_svg_layer(mapSize, tags, articles, isPreview)
  create_html_layer(mapSize, tags, articles, isPreview) 

  const loadingPlaceholder = document.querySelector('.loading-placeholder')
  loadingPlaceholder.classList.add('dn')

  // init camera
  init_camera(mapSize)

  if (!isPreview) {
    // -- origami
    if (origamiSplashGroup !== undefined) {
      randomize_map(articles, tags, null) 

      // splash: follow up and remove bgc, add opacity
      splashPost()
      
      let opacity = 1.0
      origami(origamiSplashGroup, mapSize, tags, opacity, false)
    }
    // --

    // map-nav-toggle on small screens
    mapNavToggle()
    viewToggle()
    window.addEventListener('resize', () => {
      viewToggle()
    })

    // -- map-help init
    mapHelp() 

    // -- nav, toggle switch
    navToggleSwitch(articles, tags) 

    // -- mapResultBox init
    // on node click, request search and show map-result-list
    mapListBox(articles, tags, true, false)

    // expand in Reader page
    const mapListExpand = document.querySelector('#map-list-expand-in-reader')
    mapListExpand.addEventListener('click', expandInReaderPage)

    const splitBoxExpand = document.querySelector('#split-box-expand-in-reader')
    splitBoxExpand.addEventListener('click', expandInReaderPage)
    
    // -- mapping-list buttons
    mapListBoxToolbar(articles, tags, false)

    // split-box toolbar
    splitBoxToolbar()
  } 

  // visually select all selected nodes
  if (!isPreview) {
    mark_selected_nodes(articles, tags)
  }

  const currentURL = window.location.href
  if (currentURL.split('?').length > 1) {
    let params = parseURL(currentURL)

    if (params.ids.length > 0) {
      randomize_map(articles, tags, params.ids[0])
      nodeHighlightURL(params, articles, tags)
    } else {
      // randomize map-layout upon page load
      if (origamiSplashGroup === undefined) {
        randomize_map(articles, tags, null)
      }
    }

    if (params.splitBox) {
      const mapResultBox = document.querySelector('#map-result-box')
      let splitBox = document.querySelector('#split-box') 
      const viewToggle = document.querySelector('#view-toggle')

      toggleSplitBox(mapResultBox, splitBox, viewToggle, articles, tags)
    }

  } else {
    // randomize map-layout upon page load
    if (origamiSplashGroup === undefined) {
      randomize_map(articles, tags, null)
    }

    // origami
    const origamiGroup = document.querySelector('#origami-group-ui')
    Array.from(origamiGroup.children).map(child => {
        child.remove()
    })

    let opacity = 0.5
    origami(origamiGroup, mapSize, tags, opacity, true)

  } 
}

// randomize map layout
function randomize_map(articles, tags, selectedArticle) {
  let article_random = select_article

  function getRandomInt(max) {
    return Math.floor(Math.random() * max);
  }

  if (selectedArticle === null) {
    const articleID = getRandomInt(articles.length)
    article_random = articles[articleID]

  } else {
    article_random = articles.find(article => article.id === selectedArticle)
  }

  if (article_random.html_element !== null) {

    // move articles to random location, re-compute the map
    // set new random location by taking into account map boundaries 
    let random_x = getRandomInt(8000)
    let random_y = getRandomInt(8000)
    let article_random_location = ear.vector(random_x, random_y)

    let adjusted_position = keepPositionWithinMapBoundaries(article_random.html_element, article_random_location)
    article_random.map_location = ear.vector(adjusted_position.x, adjusted_position.y)

    set_camera_position(article_random.map_location.x, article_random.map_location.y)
    compute_force_directed_layout(articles, [article_random.id])
    update_html_layer(tags, articles, false)

    let article_random_html = article_random.html_element
    article_random_html.style.zIndex = '5'

  }

}

function parseURL(currentURL) {
  const queryString = currentURL.split('?')[1].split('&')

  let params = {ids: [],
                mapList: false,
                splitBox: false
               }

  queryString.map(p => {
    const pair = p.split('=')
    if (pair[0] === 'view-in-map') {
      let ids = pair[1].split('.').map(p => Number(p))
      params.ids = ids

    } else if (pair[0] === 'map-list') {
      params.mapList = pair[1] === 'true'
        ? true
        : false

    } else if (pair[0] === 'split-box') {
      params.splitBox = pair[1] === 'true'
        ? true
        : false
    }

  })

  return params
}

function nodeHighlightURL(params, articles, tags) {
  let db = sessionDB.getAll()

  if (db.articles.length > 0) {
    if (params.mapList) {
      document.querySelector('#map-result-box').classList.remove('dn')
    } else {
      document.querySelector('#map-result-box').classList.add('dn')
    }

    params.ids.map(id => {
      
      const selectedItem = articles.find(article => article.id === id) !== undefined
            ? articles.find(article => article.id === id)
            : tags.find(tag => tag.id === id)

      if (selectedItem !== undefined) {
        set_camera_position(selectedItem.map_location.x, selectedItem.map_location.y)
        if (selectedItem.type === 'post') {
          select_article(selectedItem)
        } else if (selectedItem.type === 'tag') {
          select_tag(selectedItem)
        }
      }
    })
  }
}

function mark_selected_nodes(articles, tags) {
  let db = sessionDB.getAll()
  if (db.articles.length > 0) {
    db.articles.map(node => {

      const selectedNode = node !== undefined && node.type === 'post'
            ? articles.find(article => article.id === node.id)
            : tags.find(tag => tag.id === node.id)

      if (node.type === 'post') {
        select_article(selectedNode)
      } else if (node.type === 'tag') {
        select_tag(selectedNode)
      }
    })
  }
}

function toggleSplitBox(mapResultBox, splitBox, viewToggle, articles, tags) {
  splitButtonClick(mapResultBox, splitBox, viewToggle, articles, tags)
}

module.exports = init_map
  
