const TransformationMatrix = require("./transformation-matrix.js")

let is_dragging = false
let pointer_down_pos = {x: 0, y: 0}
let camera_down = {
  x: 0,
  y: 0,
}
let zoom_level = 0

function getPointFromEvent (event) {
  let point = {
    x: 0,
    y: 0
  }

  // if event is triggered by a touch event, we get the position of the first finger
  if (event.targetTouches) {
    point.x = event.targetTouches[0].clientX
    point.y = event.targetTouches[0].clientY
  } else {
    point.x = event.clientX
    point.y = event.clientY
  }

  return point
}

// set the transform to the identity matrix
let transform_matrix = new TransformationMatrix()

let html, svg, svg_group, svg_splash_group, mapSize
function init_camera(map_size) {
  html = document.querySelector('#map-ui')
  svg = document.querySelector('#map-graph svg')
  svg_group = document.querySelector('#svg-group-wrap')
  svg_splash_group = document.querySelector('#splash-origami-group')

  mapSize = map_size

  html.style['will-change'] = 'transform'
  svg_group.style['will-change'] = 'transform'

  if (svg_splash_group !== null) {
    svg_splash_group.style['will-change'] = 'transform'
  }

  start_panning()
  
  // --- zoom
  const map_zoom_plus = document.querySelector('#map-zoom-plus')
  const map_zoom_minus = document.querySelector('#map-zoom-minus')

  map_zoom_plus.addEventListener('click', (e) => {
    zoom_camera('plus')
  })

  map_zoom_minus.addEventListener('click', (e) => {
    zoom_camera('minus')
  })
}

function keepCameraWithinMapBoundaries() {
  // -- force camera to stick to max svg map size
  let camera_top_left = get_map_position(0, 0)
  let camera_bottom_right = get_map_position(window.innerWidth, window.innerHeight)

  let diff = {x: 0, y: 0}
  if (camera_top_left.x < -(mapSize.width / 2)) {
    diff.x = camera_top_left.x + (mapSize.width / 2)
  }
  if (camera_top_left.y < -(mapSize.height / 2)) {
    diff.y = camera_top_left.y + (mapSize.height / 2)
  }

  if (camera_bottom_right.x > (mapSize.width / 2)) {
    diff.x = camera_bottom_right.x - (mapSize.width / 2)
  }

  if (camera_bottom_right.y > (mapSize.height / 2)) {
    diff.y = camera_bottom_right.y - (mapSize.height / 2)
  }

  transform_matrix.translate(diff.x, diff.y)
}

function keepPositionWithinMapBoundaries(html, position) {
  let diff = {x: 0, y: 0}

  if ((position.x - html.offsetWidth) < -(mapSize.width / 2)) {
    diff.x = (position.x + html.offsetWidth) + (mapSize.width / 2)
  }
  if ((position.y - html.offsetHeight) < -(mapSize.height / 2)) {
    diff.y = (position.y + html.offsetHeight) + (mapSize.height /2)
  }

  if ((position.x + html.offsetWidth) > (mapSize.width / 2)) {
    diff.x = (position.x - html.offsetWidth) - (mapSize.width / 2)
  }
  if ((position.y + html.offsetHeight) > (mapSize.height / 2)) {
    diff.y = (position.y - html.offsetHeight) - (mapSize.height /2)
  }

  diff.x !== 0 ? diff.x : position.x
  diff.y !== 0 ? diff.y : position.y

  return diff
}

function update_camera() {
  keepCameraWithinMapBoundaries()

  let new_css_matrix = transform_matrix.get_css_string()
  html.style.transform = new_css_matrix
  svg_group.style.transform = new_css_matrix
  
  if (svg_splash_group !== null) {
    svg_splash_group.style.transform = new_css_matrix
  }
  
}

// panning
function start_panning() {
  if (window.PointerEvents) {
    window.addEventListener('pointerdown', onPointerDown)
    window.addEventListener('pointerup', onPointerUp)
    window.addEventListener('pointerleave', onPointerUp) // *
    window.addEventListener('pointermove', onPointerMove)
  } else {
    // add all mouse events listeners fallback
    window.addEventListener('mousedown', onPointerDown)
    window.addEventListener('mouseup', onPointerUp)
    window.addEventListener('mouseleave', onPointerUp) // *
    window.addEventListener('mousemove', onPointerMove)

    // add all touch events listeners fallback
    // this is not working on touch screens, the touch is not being recognized
    // also because it should better be touch and press hold for a few seconds?
    window.addEventListener('touchstart', onPointerDown)
    window.addEventListener('touchend', onPointerUp)
    window.addEventListener('touchmove', onPointerMove)

    if (/iPad|iPhone|iPod/.test(navigator.platform)
        || (navigator.platform === 'MacIntel' && navigator.maxTouchPoints > 1)
        && !window.MSStream) {

      // prevent pinch-to-zoom on old iOS (<= v13)
      window.document.addEventListener('touchmove', e => {
        if(e.scale !== 1) {
          e.preventDefault();
        }
      }, {passive: false});

      // prevent double-tap-to-zoom
      // <https://stackoverflow.com/a/66798697>
      let drags = new Set() //set of all active drags
      document.addEventListener("touchmove", function(event){
        if(!event.isTrusted) return //don't react to fake touches
        Array.from(event.changedTouches).forEach(function(touch){
          drags.add(touch.identifier) //mark this touch as a drag
        })
      })
      document.addEventListener("touchend", function(event){
        if(!event.isTrusted)return
        let isDrag = false
        Array.from(event.changedTouches).forEach(function(touch){
          if(drags.has(touch.identifier)){
            isDrag = true
          }
          drags.delete(touch.identifier) //touch ended, so delete it
        })
        if(!isDrag && document.activeElement == document.body){
          //note that double-tap only happens when the body is active
          event.preventDefault() //don't zoom
          event.stopPropagation() //don't relay event
          event.target.focus() //in case it's an input element
          event.target.click() //in case it has a click handler
          event.target.dispatchEvent(new TouchEvent("touchend",event))
          //dispatch a copy of this event (for other touch handlers)
        }
      })

    }

  }

  // * `{pointer,mouse}leave` helps (?) if you drag and release by letting
  //   the cursor going out the "browser's window / viewport"
}

function stop_panning() {
  if (window.PointerEvents) {
    window.removeEventListener('pointerdown', onPointerDown)
    window.removeEventListener('pointerup', onPointerUp)
    window.removeEventListener('pointerleave', onPointerUp)
    window.removeEventListener('pointermove', onPointerMove)
  } else {
    // remove all mouse events listeners fallback
    window.removeEventListener('mousedown', onPointerDown)
    window.removeEventListener('mouseup', onPointerUp)
    window.removeEventListener('mouseleave', onPointerUp)
    window.removeEventListener('mousemove', onPointerMove)

    // remove all touch events listeners fallback
    window.removeEventListener('touchstart', onPointerDown)
    window.removeEventListener('touchend', onPointerUp)
    window.removeEventListener('touchmove', onPointerMove)
  }
}

function onPointerDown(e) {
  // allow dragging node of type post
  if (e.target.dataset !== undefined && 'type' in e.target.dataset) {
    if (e.target.dataset.type === 'post-inner' || e.target.dataset.type === 'post') {
      return
    }
  }

  is_dragging = true
  let pointer_pos = getPointFromEvent(e)
  let old_camera = transform_matrix.get_position()

  const mapMenuOpen = document.querySelector('#map-menu-open')
  let scmMapListbox = false
  if (mapMenuOpen !== null) {
   scmMapListbox = mapMenuOpen.classList.contains('dn')
  }

  update_camera() 

  pointer_down_pos.x = pointer_pos.x - old_camera.x
  pointer_down_pos.y = pointer_pos.y - old_camera.y
}

function onPointerMove(e) {
  let pointer_pos = getPointFromEvent(e)

  // prevent dragging to go beyond SVG map size (width, height)
  const map_position = get_map_position(pointer_pos.x, pointer_pos.y)

  if (!is_dragging) {
    return
  }

  // when dragging map, change cursor to draggable-hand icon
  svg.classList.add('curp')

  let old_camera = transform_matrix.get_position()
  let scale = transform_matrix.get_scale().scale_x

  let camera_new = {
    x: pointer_pos.x - pointer_down_pos.x,
    y: pointer_pos.y - pointer_down_pos.y
  } 

  let delta = {
    x: (camera_new.x - old_camera.x),
    y: (camera_new.y - old_camera.y)
  } 

  const mapMenuOpen = document.querySelector('#map-menu-open')
  let scmMapListbox = false
  if (mapMenuOpen !== null) {
    scmMapListbox = mapMenuOpen.classList.contains('dn')
  }

  // block camera repainting if small screen or map nav is open
  // if (window.innerWidth > 1023 || scmMapListbox) {
    transform_matrix.translate(delta.x/scale, delta.y/scale)
    update_camera() 
  // }

}

function onPointerUp(e) {
  is_dragging = false

  // reset draggable-hand cursor
  svg.classList.remove('curp')
}


// ZOOM CAMERA
const ZOOM_SPEED = 1.2

function zoom_camera(direction) {
  // Calculate offset point in center of screen
  let offset = transform_matrix.get_inverse().transform_point({x: window.innerWidth/2, y: window.innerHeight/2})
  transform_matrix.translate(offset.x, offset.y) 

  // Zoom in or out
  if (direction === 'plus' && zoom_level < 5) {
    zoom_level++
    transform_matrix.scale(ZOOM_SPEED, ZOOM_SPEED) 

  } else if (direction === 'minus' && zoom_level > -5) {
    zoom_level--
    transform_matrix.scale(1/ZOOM_SPEED, 1/ZOOM_SPEED)
  }

  // Reset offsets
  transform_matrix.translate(-offset.x, -offset.y)

  update_camera()
}

// Compute Offsets
function get_camera_center() {
  return transform_matrix.get_inverse().transform_point({x: window.innerWidth/2, y: window.innerHeight/2})
}

function set_camera_position(x, y) {
  transform_matrix.reset()
  transform_matrix.translate(-x + window.innerWidth/2, -y + window.innerHeight/2)
  update_camera()
}

function get_map_position(x, y) {
  return transform_matrix.get_inverse().transform_point({x, y})
}

// check for wp-adminbar, adjust e.{x,y}
// could probably be solved differently
function get_cursor_position(x, y) {
  let cursor = {x: null, y: null}

  const wp_adminbar = document.querySelector('#wpadminbar')
  if (wp_adminbar) {
    cursor['x'] = x 
    cursor['y'] = y - wp_adminbar.getBoundingClientRect().height
  } else {
    cursor['x'] = x 
    cursor['y'] = y
  }

  return cursor
}

function get_camera_zoom(){
  return transform_matrix.get_scale().scale_x
}

function get_camera_zoom_reverse(){
  return transform_matrix.get_inverse().get_scale().scale_x
}

module.exports = {init_camera,
                  start_panning, stop_panning,
                  get_camera_center, get_map_position, get_cursor_position,
                  on_pointer_up: onPointerUp,
                  set_camera_position,
                  get_camera_zoom, get_camera_zoom_reverse,
                  keepPositionWithinMapBoundaries}
