"use strict";

const drawMap = require('./draw-map')
const fetchPUT = require('./fetch').put
const mapUIDelete = require('./map-ui-delete')

// draw points on map
// manipulate them with user interactions
// (hover, click, drag)
async function mapPreviewCreateCardinals(svg, points) {
  const EPSILON = 0.1;
  let graph = {};

  let dots = drawMap(svg, points, 'tag', graph, EPSILON)

  let dotSelected = null
  svg.onPress = (e) => {
    console.log('svg-press =>', e.target)
    if (!graph.vertices_coords) {
      return;
    }

    // we set the currently selected node now
    dotSelected = e.target
  };

  svg.onRelease = async(e) => {
    if (!graph.vertices_coords || dotSelected.nodeName === 'svg') {
      return;
    }

    // when a dot is relased, we want to update the tag x,y to the new values
    // we talk with WP APIs for this
    // POST /wp/v2/tags/<id>
    const tagID = Number(dotSelected.dataset.id)
    const map_location = [Number(dotSelected.getAttribute('cx')),
                          Number(dotSelected.getAttribute('cy'))];

    // we must pass tagID both in URL as well as in body
    try {
      const t = await fetchPUT(`/wp-json/wp/v2/tags/${tagID}`, {
        id: tagID,
        map_location: map_location
      })

      // check if we have added a new node, in case push it to points[]
      // so we can re-draw the map correctly
      // check if dotSelected.id matches any item in points[i].id
      // if not, we created a new tag
      const tagNew = points.find(point => point.id === Number(dotSelected.dataset.id))
      if (tagNew === undefined) {
        points.push(t)
      } else {
        // we manually update dragged node map-location x,y
        points[Number(dotSelected.dataset.idx)]['map_location'] = map_location 
      }

      // clean canvas (eg remove everything from SVG)
      svg.removeChildren()

      // clean map-ui nodes
      mapUIDelete('tag', null)

      // redraw map, pass updated points[]
      // reset graph object too
      graph = {}
      dots = drawMap(svg, points, 'tag', graph, EPSILON)

    } catch (err) {
      console.log('fetchPUT, catch err (what to do?) =>', err)
    }

  };

  svg.onMove = (e) => {
    // checking for `dotSelected === null` prevents onMove to trigger
    // when we drag quickly a node /:
    if (!graph.vertices_coords || dotSelected === null) {
      return;
    }

    // show x,y on map
    const mapCoord = document.querySelector('#map-coord')
    if (mapCoord !== null) {
      mapCoord.innerHTML = `x: ${e.x.toFixed(3)}, y: ${e.y.toFixed(3)}`
    }

    // find point index to target below
    const endpoints_vertex_equivalent = graph.vertices_coords
          .map(coords => ear.math.distance(coords, [e.x, e.y]) < EPSILON /5)
          .map((on_vertex, i) => on_vertex ? i : undefined)
          .filter(a => a !== undefined);

    // set every dot to black color
    dots.forEach((dot) => dot.fill("#889AA4"))

    // set dot[i] w/ on-hover color
    // set dotSelected to on-hover element
    const tagNew = points.find(point => point.id === Number(dotSelected.dataset.id))

    if (tagNew === undefined) {
      dots.push(dotSelected)
    }

    endpoints_vertex_equivalent.forEach(i => {
      dots[i].fill("#212121")
      console.log('EVQ =>', dots[i])
      // we make sure the dotSelected node is still the right one
      // eg the one we are dragging right now
      dotSelected = dots[i]
    })

    // -- drag node around
    if (dotSelected !== null && dotSelected.tagName === 'circle') {

      if (e.buttons === 0) {
        return
      }

      dotSelected.setPosition(e.x, e.y)

      const dotIdx = Number(dotSelected.dataset.idx)
      graph.vertices_coords[dotIdx] = [e.x, e.y]
    }

  }
};

module.exports = mapPreviewCreateCardinals
