"use strict";

const mapUILocationSync = require('./map-ui-location-sync')

function mapPanning(svg, viewBox) { 

  if (window.PointerEvents) {
    svg.addEventListener('pointerdown', onPointerDown)
    svg.addEventListener('pointerup', onPointerUp)
    svg.addEventListener('pointerleave', onPointerUp)
    svg.addEventListener('pointermove', onPointerMove)
  } else {
    // add all mouse events listeners fallback
    svg.addEventListener('mousedown', onPointerDown)
    svg.addEventListener('mouseup', onPointerUp)
    svg.addEventListener('mouseleave', onPointerUp)
    svg.addEventListener('mousemove', onPointerMove)

    // add all touch events listeners fallback
    svg.addEventListener('touchstart', onPointerDown)
    svg.addEventListener('touchend', onPointerUp)
    svg.addEventListener('touchmove', onPointerMove)
  }

  // get x & y values from the pointer event
  function getPointFromEvent (event) {
    let point = {
      x: 0,
      y: 0
    }

    // if even is triggered by a touch event, we get the position of the first finger
    if (event.targetTouches) {
      point.x = event.targetTouches[0].clientX
      point.y = event.targetTouches[0].clientY
    } else {
      point.x = event.clientX
      point.y = event.clientY
    }
    
    return point
  }

  // isPointerDown flag
  let isPointerDown = false

  // original coordinates when the user start pressing the mouse or touching the screen
  let pointerOrigin = {
    x: 0,
    y: 0
  }

  function onPointerDown(e) {
    // console.log('onPointerDown =>', event)

    // don't let the user drag the map if clicking on a map-graph-node
    if (e.target.nodeName === 'circle') {
      return
    }

    // map-ui: disable nodes from being "seen" from pointer events
    // eg if by accident dragging and ending over a map-ui-node, w/o this
    // pointer can't perform mousedown, etc
    // const mapUINodes = Array.from(document.querySelector('#map-ui').children)
    // mapUINodes.map(node => node.classList.add('pen'))

    isPointerDown = true
    
    // we get the pointer position on click/touchdown so we can get the value once the user starts to drag
    const pointerPosition = getPointFromEvent(e)
    pointerOrigin.x = pointerPosition.x
    pointerOrigin.y = pointerPosition.y
  } 

  // the distances calculated from the pointer will be stored here
  let newViewBox = {
    x: 0,
    y: 0
  }

  // calculate the ratio based on the viewbox width and the svg width
  let ratio = viewBox.width / svg.getBoundingClientRect().width
  window.addEventListener('resize', () => {
    ratio = viewBox.width / svg.getBoundingClientRect().width
  })

  // Function called by the event listeners when user start moving/dragging
  function onPointerMove (e) {
    if (!isPointerDown) {
      return
    }

    // this prevent user to do a selection on the page
    e.preventDefault()

    // get the pointer position
    const pointerPosition = getPointFromEvent(e)

    // we calculate the distance between the pointer origin and the current position
    // the viewbox x & y values must be calculated from the original values and the distances
    newViewBox.x = viewBox.x - ((pointerPosition.x - pointerOrigin.x) * ratio)
    newViewBox.y = viewBox.y - ((pointerPosition.y - pointerOrigin.y) * ratio)

    // we create a string with the new viewbox values
    // the x & y values are equal to the current viewbox minus the calculated distances
    var viewBoxString = `${newViewBox.x} ${newViewBox.y} ${viewBox.width} ${viewBox.height}`
    svg.setAttribute('viewBox', viewBoxString);
    
    // we update the map-ui nodes
    mapUILocationSync()
  }

  function onPointerUp() {
    // console.log('onPointerUp', event)

    isPointerDown = false

    // check if anyting changed between pointer-down and pointer-up
    // if yes, update map viewbox and sync map-ui, else ignore
    const hasPointerMoved = newViewBox.x !== viewBox.x || newViewBox.y !== viewBox.y

    if (hasPointerMoved) {
      // we save the viewbox coordinates based on the last pointer offsets
      viewBox.x = newViewBox.x
      viewBox.y = newViewBox.y

      mapUILocationSync()

      // map-ui-nodes: re-enable pointer events upon pointer being released
      // const mapUINodes = Array.from(document.querySelector('#map-ui').children)
      // mapUINodes.map(node => node.classList.remove('pen'))
    }
  }

}

module.exports = mapPanning
