const html = require('nanohtml')
const raw = require('nanohtml/raw')
const sessionDB = require('./session-db')
const icons = require('./map-icons')
const icon_close = require('./svg-icons').icon_panel_close
const mapNodeReset = require('./map/map-node-reset')
const mapListEdit = require('./map-list-edit')
const updateSplitReader = require('./map/split-reader-update')

function mapListBox(articles, tags, isScrolling, smallScreenDelete) { 
  if (sessionDB.exists()) {
    let db = sessionDB.getAll()

    const mapResultBox = document.querySelector('#map-result-box')
    const mapResultHeader = mapResultBox.querySelector('header')
    const mapResultList = mapResultBox.querySelector('#map-result-list') 

    if (db.articles.length > 0) {
      db.articles.map(entry => {
        let node = entry.type === 'post'
            ? articles.find(art => art.id === entry.id)
            : tags.find(tag => tag.id === entry.id)

        node.connection = entry.connection
        node.ghost = entry.ghost

        // if article.connection is not true, append article
        // and append articles.connection true below each keyword
        if (!node.connection) {
          mapListBoxAppend(node, mapResultBox, true, articles, tags, isScrolling)
        }
      })

      mapListBoxOpen(mapResultBox, smallScreenDelete)

    } else {
      // close map-list-box

      Array.from(mapResultList.children).forEach(item => {
        item.remove()
      })

      mapResultBox.classList.add('dn')
    }

  }
}

function mapListBoxOpen(mapResultBox, smallScreenDelete) {

  if (window.innerWidth > 1023 || smallScreenDelete) {
    let isCollapsed = mapResultBox.querySelector('.map-list-box')
        .classList.contains('collapsed')

    if (isCollapsed) {
      mapResultBox.querySelector('.map-list-box').classList.remove('collapsed')

      let btnClose = mapResultBox.querySelector('#map-list-close')
      btnClose.children[0].classList.remove('dn')
      btnClose.children[0].classList.add('db')

      btnClose.children[1].classList.remove('db')
      btnClose.children[1].classList.add('dn')

    } else {
      mapResultBox.classList.remove('dn')
    }

    let mapResultList = document.querySelector('#map-result-list')
    if (mapResultList.lastElementChild !== null) {
      mapResultList.lastElementChild.scrollIntoView()
    }
  }
}

function mapListBoxMinimize(mapResultBox) {
  let mapResultCollapse = mapResultBox.querySelector('.collapsible-wrapper')
  mapResultCollapse.classList.toggle('collapsed')

  // swap icon ^ / v
  let t = document.querySelector('#map-list-close')
  if (mapResultCollapse.classList.contains('collapsed')) {
    t.children[0].classList.remove('db')
    t.children[0].classList.add('dn')
    
    t.children[1].classList.remove('dn')
    t.children[1].classList.add('db')

  } else {
    t.children[0].classList.remove('dn')
    t.children[0].classList.add('db')

    t.children[1].classList.remove('db')
    t.children[1].classList.add('dn')
  }
}

function createTag(entry, mapResultList, mapResultHeader, articles, tags) {
  return html`
    <div data-id="${entry.id}" onmouseover=${onHoverIn(mapResultHeader)} onmouseout=${onHoverOut(mapResultHeader)} class="x xas db-last-child-hv psr">
      <span class="x xdc">
       ${tagConnectionLine(entry)} 
      </span>
      <a class="bgc-lait-green bgc-black-hv fgc-white pr0-5 pb0-15 pl0-5 mr0-7 b-rad-2r ft-barlow fs-button-m tdn mb0-5 h18p" href="/tag/${entry.slug}">${entry.name}</a>
      <button type="button" onclick=${itemDelete(entry, mapResultList, mapResultHeader, true, true, articles, tags)} class="item-delete dn curp svg-circle-button-close-hv mla">
        <span class="db w19p h19p">${icon_close()}</span>
      </button>
    </div>
  `

  function tagConnectionLine(entry) {
    let db = sessionDB.getAll()

    if (db.articles.length > 0) {
      const tagIdx = db.articles.findIndex(article => article.id === entry.id)

      if (db.articles[tagIdx +1] === undefined) {
        return html`
          <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M1 8.625L3.25 8.625L3.25 9.375L1 9.375L1 8.625ZM7.75 8.625L12.25 8.62499L12.25 9.37499L7.75 9.375L7.75 8.625ZM16.75 8.62499L19 8.62499L19 9.37499L16.75 9.37499L16.75 8.62499Z" fill="#6A7880"/>
            <path d="M1.375 18L1.375 -7.28522e-09L0.625 7.28522e-09L0.625 18L1.375 18Z" fill="#6A7880"/>
          </svg>
        `

      } else {

        const itemNext = db.articles[tagIdx +1]
        const isConnected = itemNext.connection !== undefined ? itemNext.connection : false

        if (isConnected) {
          return html`
            <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M0.625013 18C0.625012 17.3744 0.686347 16.7627 0.803446 16.1708L1.53919 16.3163C1.43152 16.8606 1.37501 17.4235 1.37501 18L0.625013 18ZM2.2043 12.7908C2.88864 11.7686 3.76865 10.8886 4.79084 10.2043L5.20808 10.8275C4.26735 11.4573 3.45733 12.2674 2.82753 13.2081L2.2043 12.7908ZM8.17076 8.80344C8.76269 8.68634 9.37436 8.625 10 8.625L10 9.375C9.42349 9.375 8.86056 9.43151 8.31632 9.53918L8.17076 8.80344ZM10 8.625L12.25 8.625L12.25 9.375L10 9.375L10 8.625ZM16.75 8.625L19 8.625L19 9.375L16.75 9.375L16.75 8.625Z" fill="#6A7880"/>
            </svg>
            <svg width="19" height="9" viewBox="0 0 19 9" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M0.625011 -7.24311e-07L0.625007 2.25L1.37501 2.25L1.37501 7.24311e-07L0.625011 -7.24311e-07ZM0.624998 6.75L0.624994 9L1.37499 9L1.375 6.75L0.624998 6.75Z" fill="#6A7880"/>
            </svg>
          `
        } else {
          return html`
            <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M1 8.625L3.25 8.625L3.25 9.375L1 9.375L1 8.625ZM7.75 8.625L12.25 8.62499L12.25 9.37499L7.75 9.375L7.75 8.625ZM16.75 8.62499L19 8.62499L19 9.37499L16.75 9.37499L16.75 8.62499Z" fill="#6A7880"/>
                <path d="M1.375 18L1.375 -7.28522e-09L0.625 7.28522e-09L0.625 18L1.375 18Z" fill="#6A7880"/>
            </svg>
          `
        }
      }
    }

  }

}

function createPost(entry, mapResultList, mapResultHeader, isLast, articles, tags) {
  if (entry.categories[0] !== undefined) {
    const category = entry.categories[0] !== undefined
      ? entry.categories[0].slug
      : 'default'
    const icon = icons(category)

    const itemLink = entry.ghost
          ? html`<span class="dib fgc-dark-gray tdn w78 pb0-5">${entry.title}</span>`
          : html`<a href="${entry.slug}" class="dib fgc-black fgc-lait-green-hv tdn w78 pb0-5">${entry.title}</a>`

    return html`
      <div data-id="${entry.id}" onmouseover=${onHoverIn(mapResultHeader, entry.ghost)} onmouseout=${onHoverOut(mapResultHeader)} class="x xas db-last-child-hv fs-md-small">
        ${connectionLine(entry.title, entry.connection, isLast)}
        <span class="dib mr0-7 h18p">${icon}</span>
        ${itemLink}
        ${removeItem(entry.ghost, articles, tags)} 
      </div>
    `

    function connectionLine(title, hasConnection, isLast) {
      if (hasConnection) {
        if (isLast) {
          return html`
            <span class="x xdc">
              <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M0.625013 -5.59368e-07C0.625012 0.625632 0.686347 1.23731 0.803446 1.82923L1.53919 1.68368C1.43152 1.13944 1.37501 0.576504 1.37501 5.5937e-07L0.625013 -5.59368e-07ZM2.2043 5.20916C2.88864 6.23135 3.76865 7.11136 4.79084 7.7957L5.20808 7.17247C4.26735 6.54267 3.45733 5.73265 2.82753 4.79192L2.2043 5.20916ZM8.17076 9.19656C8.76269 9.31366 9.37436 9.375 10 9.375L10 8.625C9.42349 8.625 8.86056 8.56849 8.31632 8.46082L8.17076 9.19656ZM10 9.375L12.25 9.375L12.25 8.625L10 8.625L10 9.375ZM16.75 9.375L19 9.375L19 8.625L16.75 8.625L16.75 9.375Z" fill="#6A7880"/>
              </svg> 
            </span>
          `
        } else { 
          return html`
            <span class="x xdc">
              <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M1.375 -2.96076e-07L1.375 2.25L0.625 2.25L0.624998 2.96076e-07L1.375 -2.96076e-07ZM1.375 6.75L1.37501 11.25L0.625007 11.25L0.625004 6.75L1.375 6.75ZM1.37501 15.75L1.37501 18L0.625013 18L0.625011 15.75L1.37501 15.75Z" fill="#6A7880"/>
                <path d="M1 8.625L3.25 8.625L3.25 9.375L1 9.375L1 8.625ZM7.75 8.625L12.25 8.62499L12.25 9.37499L7.75 9.375L7.75 8.625ZM16.75 8.62499L19 8.62499L19 9.37499L16.75 9.37499L16.75 8.62499Z" fill="#6A7880"/>
              </svg>
              <svg width="19" height="9" viewBox="0 0 19 9" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M0.625011 -7.24311e-07L0.625007 2.25L1.37501 2.25L1.37501 7.24311e-07L0.625011 -7.24311e-07ZM0.624998 6.75L0.624994 9L1.37499 9L1.375 6.75L0.624998 6.75Z" fill="#6A7880"/>
              </svg>
              ${vertical(title)}
            </span>
          `
        }

      } else {

        // single item
        return html`
          <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M1 8.625L3.25 8.625L3.25 9.375L1 9.375L1 8.625ZM7.75 8.625L12.25 8.62499L12.25 9.37499L7.75 9.375L7.75 8.625ZM16.75 8.62499L19 8.62499L19 9.37499L16.75 9.37499L16.75 8.62499Z" fill="#6A7880"/>
            <path d="M1.375 18L1.375 -7.28522e-09L0.625 7.28522e-09L0.625 18L1.375 18Z" fill="#6A7880"/>
          </svg>
        `

      }
    }
    
    function vertical(text) {
      if (text.length > 75) {
        return html`
          <span class="x xdc">
            <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M1.375 -2.96076e-07L1.375 2.25L0.625 2.25L0.624998 2.96076e-07L1.375 -2.96076e-07ZM1.375 6.75L1.37501 11.25L0.625007 11.25L0.625004 6.75L1.375 6.75ZM1.37501 15.75L1.37501 18L0.625013 18L0.625011 15.75L1.37501 15.75Z" fill="#6A7880"/>
            </svg>
            <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M1.375 -2.96076e-07L1.375 2.25L0.625 2.25L0.624998 2.96076e-07L1.375 -2.96076e-07ZM1.375 6.75L1.37501 11.25L0.625007 11.25L0.625004 6.75L1.375 6.75ZM1.37501 15.75L1.37501 18L0.625013 18L0.625011 15.75L1.37501 15.75Z" fill="#6A7880"/>
            </svg>
          </span>
        `
      } else if (text.length > 39) {
        return html`
          <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M1.375 -2.96076e-07L1.375 2.25L0.625 2.25L0.624998 2.96076e-07L1.375 -2.96076e-07ZM1.375 6.75L1.37501 11.25L0.625007 11.25L0.625004 6.75L1.375 6.75ZM1.37501 15.75L1.37501 18L0.625013 18L0.625011 15.75L1.37501 15.75Z" fill="#6A7880"/>
          </svg>
        `
      }
    }

    function removeItem(ghost, articles, tags) {
      if (!ghost) {
        return html`
          <button type="button" onclick=${itemDelete(entry, mapResultList, mapResultHeader, true, true, articles, tags)} class="item-delete dn curp svg-circle-button-close-hv mla">
            <span class="db w19p h19p">${icon_close()}</span>
          </button>
        `
      }
    }
  }
}

function itemDelete(entry, mapResultList, mapResultHeader, isClickHover, removeFromDB, articles, tags) {
  if (isClickHover) {
    return () => {
      deleteNode(entry, mapResultList, mapResultHeader, removeFromDB, articles, tags)
    }
  } else {
    deleteNode(entry, mapResultList, mapResultHeader, removeFromDB, articles, tags)
  }

  function deleteNode(entry, mapResultList, mapResultHeader, removeFromDB, articles, tags) {

    if (entry.type === 'tag') {
      // de-select article selected by tag, only if not already
      // selected by clicking on the article itself
      // eg if article.id exists more than once in the list,
      // do not de-select it

      let db = sessionDB.getAll()
      entry.articles.map(art => {
        let instances = db.articles.filter(item => item.id === art.id)

        if (instances.length === 1) {
          mapNodeReset(art, false)

          // remove item from sessionDB
          if (removeFromDB) {
            sessionDB.removeItemAll(art.id)
          }
        }
      })
    }

    // deselect node
    mapNodeReset(entry, false)

    // remove item from sessionDB
    if (removeFromDB) {
      sessionDB.removeItemAll(entry.id)
    }

    // rebuild map-list
    const mapResultBox = document.querySelector('#map-result-box')
    Array.from(mapResultList.children).map(item => item.remove())

    let smallScreenDelete = !mapResultBox.classList.contains('dn')
    mapListBox(articles, tags, false, smallScreenDelete)

    // check if in Edit mode (on touch screens)
    // if yes, re-enable delete buttons next to items
    if (window.innerWidth <= 1023) {
      mapListEdit({target: null, action: 'edit'})
    }

    // if removing last item in the list, close map-list box 
    let db = sessionDB.getAll()
    if (db.articles.length === 0) {
      mapResultHeader.firstElementChild.innerText = "Mapping result"
      mapResultHeader.lastElementChild.classList.remove('vh') 

      mapResultBox.classList.add('dn')

      if (smallScreenDelete) {
        const viewToggle = document.querySelector('#view-toggle')
        viewToggle.classList.remove('x')
        viewToggle.classList.add('dn')
      }
    }

    // rebuild split-box if open
    updateSplitReader(tags, articles)

  }
}

function onHoverIn(mapResultHeader, ghost) {
  return () => {
    if (!ghost) {
      mapResultHeader.firstElementChild.innerText = "Delete this contribution from the Reader?"
      mapResultHeader.lastElementChild.classList.remove('x')
      mapResultHeader.lastElementChild.classList.add('dn')
    }
  }
}

function onHoverOut(mapResultHeader) {
  return () => {
    mapResultHeader.firstElementChild.innerText = "Mapping result"
    mapResultHeader.lastElementChild.classList.remove('dn')
    mapResultHeader.lastElementChild.classList.add('x')
  }
}

function mapListBoxAppend(node, mapResultBox, init, articles, tags, isScrolling) {
  let db = sessionDB.getAll()

  const mapResultHeader = mapResultBox.querySelector('header')
  let mapResultList = mapResultBox.querySelector('#map-result-list')
  const loading = loadingPlaceholder(mapResultBox, mapResultList)

  if (node.type === 'tag') {    
    let new_tag = {...node}

    let tag_articles = node.articles.map(article => article.id)
    new_tag.articles = tag_articles

    if (!init) {
      db.articles.push(new_tag)
      sessionDB.save(db)
    }
    
    // fetch tag connections
    const tag_connections = node.articles.map((article, idx) => {

      let new_article = {...article}
      new_article.ghost = false 
      new_article.connection = true
      delete new_article.tags

      // check if already present in map-box-list
      // and make it "grayed-out"
      const articleDuplicates = Array.from(mapResultList.children)
            .filter(art => Number(art.dataset.id) === new_article.id)

      if (articleDuplicates.length > 0) {
        new_article.ghost = true
      }

      if (!init) {
        db.articles.push(new_article)
        sessionDB.save(db)
      }

      return new_article

    })

    // -- append tag and connected articles to map-list
    const tagItem = createTag(node, mapResultList, mapResultHeader, articles, tags)
    mapResultList.append(tagItem)

    // fetch all articles added to db.articles that are part of node.articles
    // (eg tag connections) and add them to map-list
    let dbArticlesIDs = db.articles.map(article => article.connection && article.id)
    let connections = tag_connections.filter(article => dbArticlesIDs.includes(article.id))

    connections.map((article, idx) => {
      const isLast = connections.length -1 === idx
      const item = createPost(article, mapResultList, mapResultHeader, isLast, articles, tags)
      mapResultList.append(item)
    })
    
    
    loading.remove()
    
  } else if (node.type === 'post') { 

    // save article to sessionDB
    let new_article = {...node}
    new_article.connection = false
    new_article.ghost = false
    delete new_article.tags

    if (!init) {
      db.articles.push(new_article)
      sessionDB.save(db)
    }

    // create selected article and append to DOM
    let fIdx
    const filterOut = db.articles.some((item, i) => {
      fIdx = i
      return item.id === node.id
    })

    if (filterOut) {
      node.connection = false
      node.ghost = false
      const isLast = fIdx === db.articles.length -1

      const pointItem = createPost(node, mapResultList, mapResultHeader, isLast, articles, tags)
      mapResultList.append(pointItem)
    }

    loading.remove()
  }

  let isCollapsed = mapResultBox.querySelector('.map-list-box')
      .classList.contains('collapsed')
  
  if (!isCollapsed) {
    if (isScrolling && mapResultList.lastElementChild !== null) {
      mapResultList.lastElementChild.scrollIntoView()
    }
  }
}

function loadingPlaceholder(mapResultBox, mapResultList) {
  // create box
  const splitBox = document.querySelector('#split-box')
  let isSplitBoxOpen = null
  if (splitBox !== null) {
    isSplitBoxOpen = !splitBox.classList.contains('dn')
  }

  // display map-box-list only on big-enough screens
  // if =< 1024 and resizing window, close it (ah!)
  if (window.innerWidth > 1023) {
    if (isSplitBoxOpen !== null && !isSplitBoxOpen) {
      mapResultBox.classList.remove('dn')
    }
  } else {
    mapResultBox.classList.add('dn')
  }

  // put loading...
  const loading = document.createElement('p')
  loading.innerHTML = "Loading..."

  mapResultList.append(loading)

  return loading
}

function mapListBoxVisibilyRules() {
  let db = sessionDB.getAll()
  const mapResultBox = document.querySelector('#map-result-box')
  const menuOpen = document.querySelector('#map-menu-open')

  if (db.articles.length > 0) {
    if (window.innerWidth > 1023) {
      mapResultBox.classList.remove('dn')
    } else if (menuOpen.classList.contains('dn')) {
      mapResultBox.classList.add('dn')
    }
  }
}

module.exports = {mapListBox, mapListBoxOpen, mapListBoxMinimize, mapListBoxAppend,
                  createTag, createPost,
                  itemDelete,
                  mapListBoxVisibilyRules}
