"use strict";

const {WPpagination} = require('./map/fetch-data.js')
const sessionDB = require('./session-db')
const html = require('nanohtml')
const raw = require('nanohtml/raw')
const icons = require('./map-icons')
const iconPanelBack = require('./svg-icons').icon_panel_back
const iconPanelShare = require('./svg-icons').icon_panel_share
const iconPanelClose = require('./svg-icons').icon_panel_close
const generateURL = require('./map-list-link').generateURL
const set_camera_position = require('./map/camera').set_camera_position
const {select_article, select_tag} = require('./map/map-node-select')
const {navToggleMap} = require('./nav-toggle-buttons')
const mapNodeReset = require('./map/map-node-reset')
const {generateLink} = require('./map-list-link')
const {footnotesCleanup, setFootnotes} = require('./post')

function createSplitBoxItem(splitBox, article, idx) {
  return html`
    <div class="pt1 pb1 ${idx.total === idx.curr ? '' : 'bb1-dash-dark-green'}">
      ${item(splitBox, article)}
    </div>
  `
}

function item(splitBox, item) { 

  if (item.type === 'tag') {
    return html`
      <button type="button" onclick=${viewTag(splitBox, item.id, item.map_location)} class="curp fgc-white tdn bgc-lait-green bgc-black-hv pr0-5 pl0-5 pb0-15 mr0-7 b-rad-2r ft-barlow fs-md-h3">${item.name}</button>
    `

  } else if (item.type === 'post') {

    if (item.categories.length > 0) {
      let category = item.categories[0] !== undefined
            ? item.categories[0].slug
            : 'default'
      let icon = icons(category)

      let date = item.date.split('T')[0].split('-').reverse().join('.')

      return html`
        <div>
          <p class="fs-md-small pb0-5">${date}</p>
          <header class="x xdr xjb">
            <button type="button" onclick=${viewArticle(splitBox, item.id, item.map_location)} class="curp tal ft-barlow fgc-lait-green-hv fs-md-h3 pb0-5">${item.title}</button>
            <span class="dib ml0-7 h18p">${icon}</span>
          </header>
          <div class="fs-md-small pb1">${getContributors(item.contributor)}</div>
          <div class="fs-md-small copy-reset-margin">${raw(item.excerpt)}</div>
        </div>
      `
    }
  }
}

function viewArticle(splitBox, ID, map_location) {
  return async() => {

    // fetch article full content
    let article = require('../posts.json').find(art => art.id === ID)

    // fetch article connections by tag
    const article_connections = article.tags.map(tag_id => {
      let conns = require('../tags.json').filter(tag => tag === tag_id)
      return conns
    })

    // flat nested array list
    let connections = []
    article_connections.map(block => {
      block.map(item => {
        // filter out current article from connection-list
        if (item.id !== ID) {
          connections.push(item)
        }
      })
    })

    // filter out duplicates from connection-list (this is weird, but alas)
    const connections_ids = connections.map(conn => conn.id)
    const connections_filtered = connections.filter((item, index) => {
      if (!connections_ids.includes(item.id, index +1)) {
        return item
      }
    })

    // remove everything from split-box
    let currentView = document.querySelector('#split-box')
    Array.from(currentView.children).map(block => block.remove())

    let content = raw(article.content.rendered)

    // remove newline and such from the DOM
    // <https://www.sitepoint.com/removing-useless-nodes-from-the-dom/>
    const cleanNodeTypes = [3, 8]
    content = Array.from(content)
      .map(block => {
        if (block !== undefined && !cleanNodeTypes.includes(block.nodeType)) {
          block.classList.remove('bg-pr5-2', 'lg-pr0', 'bg-pl5-2', 'lg-pl0')
          return block
        }
      })

    let data = {
      type: article.type,
      title: article.title.rendered,
      id: article.id,
      url: article.link,
      contributor: article.contributor,
      date: article.date,
      categories: article._embedded['wp:term'][0],
      content: content,
      map_location: map_location,
      map_preview_image: article.map_preview_image,
      html_element: document.querySelector(`#map-ui > [data-id='${ID}']`),
      connections: connections_filtered
    }

    // and replace it with selected article
    let articleView = getArticle(splitBox, data)
    currentView.appendChild(articleView)

    const {sups, refs} = footnotesCleanup(articleView)
    setFootnotes(refs, sups, false)

    // highglight article node on the map
    // and remove any existing one
    resetHighlightNodesOnMap()  
    highlightNodeOnMap(data)
    
  }

  function getArticle(splitBox, article) {
    const date = article.date.split('T')[0].split('-').reverse().join('.')
    const category = article.categories[0] !== undefined
          ? article.categories[0].slug
          : 'default'
    const icon = icons(category)

    splitBox.scrollIntoView()

    return html`
      <div>
        <header class="pb3">
          <div class="w100 x xdr xw xab">
            <h2 class="tb-mw-copy-post bg-mw-copy-post ft-barlow fs-md-h2 w90 pb1">${article.title}</h1>
            <div class="mla">
              <button type="button" onclick=${backToReader(splitBox, article)} class="curp svg-circle-button-back-hv w19p h19p mr0-25"><span>${iconPanelBack()}</span></button>
              <button type="button" onclick=${closeReader(splitBox, article)} class="curp svg-circle-button-close-hv w19p h19p"><span>${iconPanelClose()}</span></button>
            </div>
          </div>
          <div class="psr w100 x xdc xw pb1">
            <div class="fs-sb">
              ${getContributors(article.contributor)}
            </div>
            <p class="fs-sb">${date}</p>
          </div>
          <div class="psr w100 x xdr xw">
            <span class="mla dib mr0-7 h18p">${icon}</span>
            <button type="button" onclick=${viewInMap(splitBox, article)} class="button-s mr0-5">View in map</button>
            <a href="${article.url}" class="button-s tdn mr0-5 ml0-25">Expand in Reader</a>
            <div class="dn map-result-link psa r0 bgc-black fgc-white pt0-15 pr0-25 pb0-25 pl0-25 ft-barlow fs-button-s b-rad-3 z5">URL copied!</div>
            <button type="button" id="sb-map-list-link" onclick=${mapListShare(['page', article.url])} class="w19p h19p curp svg-circle-button-share-hv ml0-25">${iconPanelShare()}</button>
          </div>
        </header>
        <section class="fs-copy-sb content-blocks-wrapper">${article.content}</section>
        <section class="bg-mw-copy-wrap">
          <h2 class="ft-barlow fs-md-h2 bb0-75-dark-green pb1">Connected Material</h2>
          ${displayConnections(article.connections)}
        </section>
      </div>
    `
  }
}

function viewTag(splitBox, ID, map_location) {
  return async() => {

    // fetch tag full content
    let tag = require('../tags.json').find(tag => tag.id === ID)
    let tag_connections = require('../posts.json').filter(post => {
      if (post.tags.includes(ID)) {
        return post
      }
    })

    // remove everything from split-box
    let currentView = document.querySelector('#split-box')
    Array.from(currentView.children).map(block => block.remove())

    let data = {
      type: 'tag',
      title: tag.name,
      id: tag.id,
      url: tag.link,
      map_location: map_location,
      html_element: document.querySelector(`#map-ui > [data-id='${ID}']`),
      svg_elements: [document.querySelector(`#svg-group-ui > [data-id='${ID}']`)],
      connections: tag_connections
    }

    // and replace it with selected article
    let tagView = getTag(splitBox, data)

    currentView.appendChild(tagView)

    // highglight article node on the map
    // and remove any existing one
    resetHighlightNodesOnMap()  
    highlightNodeOnMap(data)
    
  }

  function getTag(splitBox, tag) {
    splitBox.scrollIntoView()

    return html`
      <div>
        <header class="pb3">
          <div class="w100 x xdr xw xab pb2-5">
            <h2 class="tb-mw-copy-post bg-mw-copy-post ft-barlow fs-md-h2 w90">Keyword: <span class="tdn dib pr1 pb0-15 pl1 mr0-15 bgc-lait-green b-rad-2r ft-barlow fgc-white">${tag.title}</span></h1>
            <div class="mla">
              <button type="button" onclick=${backToReader(splitBox, tag)} class="curp svg-circle-button-back-hv w19p h19p mr0-25"><span>${iconPanelBack()}</span></button>
              <button type="button" onclick=${closeReader(splitBox, tag)} class="curp svg-circle-button-close-hv w19p h19p"><span>${iconPanelClose()}</span></button>
            </div>
          </div>
          <div class="psr w100 x xdr xw">
            <div class="mla x xdr xw">
              <button type="button" onclick=${viewInMap(splitBox, tag)} class="button-s mr0-5 ml0-25">View in map</button>
              <a href="${tag.url}" class="button-s tdn mr0-5 ml0-25">Expand in Reader</a>
              <div class="dn map-result-link psa r0 bgc-black fgc-white pt0-15 pr0-25 pb0-25 pl0-25 ft-barlow fs-button-s b-rad-3 z5">URL copied!</div>
              <button type="button" id="sb-map-list-link" onclick=${mapListShare(['page', tag.url])} class="w19p h19p curp svg-circle-button-share-hv ml0-25">${iconPanelShare()}</button>
            </div>
          </div>
        </header>
        <section class="bg-mw-copy-wrap">
          <h2 class="ft-barlow fs-md-h2 bb0-75-dark-green pb1">Connected Material</h2>
          ${displayConnections(tag.connections)}
        </section>
      </div>
    `
  } 
}

function displayConnections(data) {
  if (data.length > 0) {
    return data.map(item => {
      let categories = item._embedded['wp:term'][0]
      let category = categories[0] !== undefined
            ? categories[0].slug
            : 'default'
      let icon = icons(category)

      let date = item.date.split('T')[0].split('-').reverse().join('.')
      let contributors = item.contributor.length > 0 ? item.contributor : []

      return html`
        <div class="w100 x xdc xw pt1 pb1 connected-item bb0-75-dash-dark-green-nlc">
          <p class="w100 fs-md-small pb0-5">${date}</div>
          <header class="w100 x xdr xjb">
            <a href="${item.slug}" class="db ft-barlow fs-md-h2 bg-fs-md-h3 fgc-lait-green-hv pb0-5 tdn">${item.title.rendered}</a>
            <span class="dib ml0-7 h18p">${icon}</span>
          </header>
          <div class="w100 fs-md-small pb1">${getContributors(contributors)}</div>
          <div class="w100 fs-md-small copy-reset-margin">${raw(item.excerpt.rendered)}</div>
        </div>
      `
    })

  } else {
    return html`<p class="pt1">No post was found for this contributor yet.</p>`
  }
}

function getContributors(data) {
  if (data.length > 0) {
    return data.map(contributor => {
      return html`
        <a href="${contributor.post_name}" class="tdn fgc-lait-green-hv post-contributor-comma">${contributor.post_title}</a>
      `
    })
  }
}

function viewInMap(splitBox, node) {
  return () => {
    // close split-box
    Array.from(splitBox.children).map(block => block.remove())

    let layout = splitBoxLayout(splitBox)
    splitBox.appendChild(layout)
    splitBox.classList.add('dn')
    
    // center map to article
    set_camera_position(node.map_location[0], node.map_location[1])

    // switch to correct nav-toggle
    navToggleMap()
  }
}

function backToReader(splitBox, item) {
  return () => {
    // remove everything from split-box (article-view)
    Array.from(splitBox.children).map(block => block.remove())

    // and put back split-box basic view
    let layout = splitBoxLayout(splitBox)
    splitBox.appendChild(layout)

    populateSplitBox(splitBox)

    // reset selected node on the map
    mapNodeReset(item, true)
  } 
}

function populateSplitBox(layout) {

  let loading = document.createElement('p')
  loading.innerHTML = "Loading..."
  layout.append(loading)

  // populate w/ articles, if any
  if (sessionDB.exists()) {
    const db = sessionDB.getAll()

    if (db.articles.length > 0) {
      loading.remove()

      db.articles.map((article, i) => {
        let idx = {total: db.articles.length -1, curr: i}
        const item = createSplitBoxItem(layout, article, idx)
        layout.append(item)
      })

    }
  }
}

function splitBoxLayout(splitBox) {
  return html`
    <header class="psr x xdr xw xac pb1 bb1-dark-green">
      <h2 class="ft-barlow fs-md-h2">Reader</h2>
      <a id="split-box-expand-in-reader" href="/reader?" class="button-s tdn mla mr0-5">Expand in Reader</a>
      <button type="button" onclick=${closeReader(splitBox, null)} style="top: -1.5rem" class="button-s mr0-5">Close</button>
      <div style="top: -1.5rem" class="dn map-result-link psa r0 bgc-black fgc-white pt0-15 pr0-25 pb0-25 pl0-25 ft-barlow fs-button-s b-rad-3 z5">Reader URL copied!</div>
      <button type="button" id="sb-map-list-link" onclick=${mapListShare(['reader'])} class="w19p h19p curp svg-circle-button-share-hv ml0-25">${iconPanelShare()}</button>
    </header>
    <div id="split-box-list"></div>
  `
}

function mapListShare(context) {
  return () => {
    let btn = document.querySelector('#sb-map-list-link')
    let mapResultLink = btn.previousElementSibling
    generateLink(btn, mapResultLink, context)
  }
}

function closeReader(splitBox, node) {
  return () => {
    // clear all articles and close box
    Array.from(splitBox.children).forEach(item => {
      item.remove()
    })

    // and put back split-box basic view
    let layout = splitBoxLayout(splitBox)
    splitBox.appendChild(layout)

    splitBox.classList.add('dn')

    // reset selected node on the map
    if (node !== null) {
      mapNodeReset(node, true)
    }

    navToggleMap()

  }
}

function highlightNodeOnMap(node) {
  // we need to decenter the node position because on the right side
  // we have the split-box open
  // we divide window's width by 4 as the take the left side first,
  // then we half it so to center the node in it
  const PADDING = 16 * 1.5
  let shiftX = (window.innerWidth / 4) - PADDING

  if (node.type === 'post') {
    select_article(node)
    node.html_element.classList.remove('bs-bk')
    node.html_element.classList.add('bs-glow', 'z4')

    set_camera_position((node.map_location[0] + shiftX), node.map_location[1])

  } else if (node.type === 'tag') {
    select_tag(node)
    node.html_element.classList.remove('bs-bk')
    node.html_element.classList.add('bs-glow', 'z4')

    // node tag has a label above the node, so we should:
    // - get the node's total height
    // - divide that by half and substract half of the circle height
    //   (which is the map_location.y current center-point)
    let circleGBR = node.svg_elements[0].getBoundingClientRect()
    let labelGBR = node.html_element.getBoundingClientRect()
    let circleHeight = circleGBR.height
    let labelHeight = labelGBR.height
    let nodeHeight = circleGBR.bottom - labelGBR.top
    let shiftY = (nodeHeight / 2) - (circleHeight / 2)

    set_camera_position((node.map_location[0] + shiftX), node.map_location[1] - shiftY)

  }
}

function resetHighlightNodesOnMap() {
  const nodes = document.querySelectorAll("[data-type]")
  Array.from(nodes).forEach(node => node.classList.remove('bs-glow', 'z4'))
}

function makeSplitReaderList(tags, articles) {
  let db = sessionDB.getAll()

  let splitBox = document.querySelector('#split-box') 
  let splitBoxExpandInReader = splitBox.querySelector('#split-box-expand-in-reader')
  let splitBoxList = splitBox.querySelector('#split-box-list') 

  db.articles
    .filter(article => !article.ghost)
    .map((article, i) => {

      let selected_node = article.type === 'post'
          ? articles.find(art => art.id === article.id)
          : tags.find(tag => tag.id === article.id)

      let idx = {total: db.articles.length -1, curr: i}
      const item = createSplitBoxItem(splitBox, selected_node, idx)
      splitBoxList.append(item)
    })

  splitBox.scroll(0, 0)
}

module.exports = {createSplitBoxItem,
                  splitBoxLayout, populateSplitBox,
                  makeSplitReaderList,
                  closeReader}
