const html = require('nanohtml')
const {WPpagination} = require('./map/fetch-data.js')

async function create_map() {
  //Load all data from wordpress API
  let tags = await load_tags()
  let articles = await load_articles(tags)
  console.log("tags -> ", tags[0].map_location.x);
  console.log("articles -> ", articles);

  //Create SVG & HTML element layers
  let viewBox = compute_view_box()
  create_svg_layer(viewBox, tags, articles)
  create_html_layer(viewBox, tags, articles)
}

async function load_tags(){
  const fetchURL = '/wp-json/wp/v2/tags?hide_empty=1&per_page=100'
  let tags = await WPpagination(fetchURL)
      .catch(err => {
        console.log('err WP-pagination =>', err)
      })

  //Map locations to ear vectors 
  //Sort by ID ASC order
  tags = tags.map(t=>{
    t.map_location = ear.vector(t.map_location[0], t.map_location[1])
    return t
  }).sort((a, b) => {
    return a.id - b.id;
  })

  return tags
}

async function load_articles(tags){
  const fetchURL = '/wp-json/wp/v2/posts?_embed&per_page=100'
  let articles = await WPpagination(fetchURL)
      .catch(err => {
        console.log('err WP-pagination =>', err)
      })

    articles = articles.map((article) => {
      //Lookup tag objects and bind them to the article
      const article_tags = article._embedded['wp:term'][1].map(tag=>{
        return tags.find(t=>t.id == tag.id)
      })

      return {
        id: article.id,
        title: article.title.rendered,
        slug: article.slug,
        author: article.twla_author ? article.twla_author.post_title : null,
        excerpt: article.excerpt.rendered,
        date: article.date,
        type: article.type,
        format: article.format,
        categories: article.categories,
        tags: article_tags,
        image: 'wp:featuredmedia' in article._embedded
          ? article._embedded['wp:featuredmedia'][0]
          : null,
        count: 0,
        map_location: ear.vector(0, 0),
        post_location: article.post_location
      }
    })
    return articles
}



function create_svg_layer(viewBox, tags, articles) {
  //SETUP SVG ELEMENT
  const map = document.querySelector('#map-graph');
  const svg = ear.svg(map, (svg) => {
    svg
      .setAttributes({width: window.innerWidth, height: window.innerHeight})
      .size(viewBox.width, viewBox.height)
      //.setAttributes({id: 'map-wrapper', transform: `matrix(${matrix.join(' ')})`})
      .addClass('ba1-salmon')
  });

  //CREATE DOTS FOR EACH TAG
  tags.map((tag, i) => {
    svg.circle(tag.map_location)
      .radius(1)
      .fill('#6A7880')
      .addClass(`node-map node-tag circle-${tag.id} curp`)
      //.setAttributes({'data-idx': i, 'data-id': tag.id})
  })

  //CREATE LINES BETWEEN ARTICLES & TAGS
  articles.map((article, i)=>{
    article.tags.map(tag=>{
      svg.line(
        article.map_location.x, article.map_location.y,
        tag.map_location.x, tag.map_location.y
      )
      .strokeWidth(0.1)
      .stroke('#6A7880')
      //.setAttributes({'data-idx': i, 'data-id': tag.id})
    })
  })

  return svg
}

function create_html_layer(viewBox, tags, articles) {
  const mapUI = document.querySelector('#map-ui')

  //CREATE LABEL FOR EACH TAG
  tags.map((tag, i)=>{
    const htmlElement = html`
      <div data-id=${tag.id}" data-type="keyword" class="psa vh z1 curp usn tdn">
        <div class="fgc-white bgc-lait-green bgc-black-hv pr0-5 pl0-5 pb0-15 b-rad-2r ft-barlow">${tag.name}</div>
        <div class="dn psa"></div>
      </div>
    `
    mapUI.appendChild(htmlElement)

    //Compute On screen position
    let onscreenPosition = viewBox.computePosition(tag.map_location)

    //Read boundingBox and then remove vh class
    let boundingBox = htmlElement.getBoundingClientRect()
    htmlElement.classList.remove('vh')

    //Set On screen width and Height
    htmlElement.style.left = `${onscreenPosition.x - boundingBox.width/2}px`
    htmlElement.style.top  = `${onscreenPosition.y - 50}px`
    htmlElement.style.width = `${boundingBox.width}px`
    htmlElement.children[0].style.width = `${boundingBox.width}px`
  })


  //CREATE ARTICLE BOXES
  // arbitraty max-width in px (21rem)
  const MAX_WIDTH = 336 
  articles.map((article)=> {
    const htmlElement = html`<div data-id="${article.id}" data-type="post" class="psa vh z1 curp fgc-black usn tdn tac pt0-5 pr0-5 pb0-5 pl0-5 b-rad-2 fs-map-md-small">${article.title}</div>`
    mapUI.appendChild(htmlElement)
    htmlElement.classList.remove('vh')

    htmlElement.style.width = `${MAX_WIDTH}px`
    console.log(htmlElement);
  })

}

module.exports = create_map
